//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRBSpline.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Apr/02/1999:	Copied from eldBSpline.C
//
// Bugs:
//
//-------------------------------------------------------------------=72


#include "RRBSpline.h"

#include <stdio.h>

RRBSpline::RRBSpline(const int degree)
:mDegree(degree),
 mKnot(0),
 mCoeff(0)
{}

RRBSpline::RRBSpline(const int degree,const RRArray< double >& knot)
:mDegree(degree),
 mKnot(knot),
 mCoeff(0)
{}

RRBSpline::RRBSpline(
			const int degree,
			const RRArray< double >& knot,
			const RRArray< double >& coeff)
:mDegree(degree),
 mKnot(knot),
 mCoeff(coeff)
{
	assert(mKnot.getSize()-1+mDegree==mCoeff.getSize());
}

RRBSpline::RRBSpline(const RRBSpline& x)
:mDegree(x.mDegree),
 mKnot(x.mKnot),
 mCoeff(x.mCoeff)
{
	assert(mKnot.getSize()-1+mDegree==mCoeff.getSize());
}

RRBSpline::~RRBSpline()
{}

const RRBSpline& RRBSpline::operator=(const RRBSpline& x)
{
	mDegree=x.mDegree;
	mKnot=x.mKnot;
	mCoeff=x.mCoeff;
	return *this;
}

RRStatus RRBSpline::setKnot(const RRArray< double >& knot)
{
	mKnot=knot;
	mCoeff.setSize(0);
	return RR_SUCCESS;
}

RRStatus RRBSpline::setCoeff(const RRArray< double >& coeff)
{
	if(coeff.getSize()!=mKnot.getSize()-1+mDegree)
		return RR_ERROR;

	mCoeff=coeff;
	return RR_SUCCESS;
}

RRStatus RRBSpline::mfMakeMatrix(
			const RRArray< RRVec2 >& p,
			const int dim,
			double *a,
			double *b)const
{
	int nd=p.getSize();

	for(int j=0;j<dim;j++)
	{
		for(int i=j;i<dim;i++)
		{
			double s=0.;
			for(int n=0;n<nd;n++)
			{
				double nj=mfGetBase(j-mDegree,mDegree,p[n].getX());
				double ni=mfGetBase(i-mDegree,mDegree,p[n].getX());
				s+=nj*ni;
			}
			a[j*dim+i]=a[i*dim+j]=s;	
		}
		
		double z=0.;
		for(int n=0;n<nd;n++)
		{
			double nj=mfGetBase(j-mDegree,mDegree,p[n].getX());
			z+=nj*p[n].getY();
		}
		b[j]=z;	
	}

	/*
	FILE *fp=fopen("matrix.txt","w");
	fprintf(fp,"matrix\n");
	fprintf(fp,"a =\t");
	for(int m=0;m<dim;m++)
	{
		fprintf(fp,"\t");
		for(int n=0;n<dim;n++)
			fprintf(fp,"%lf ",a[m*dim+n]);
		fprintf(fp,"\n");
	}
	fprintf(fp,"b =\t");
	for(m=0;m<dim;m++)
		fprintf(fp,"%lf ",b[m]);
	fprintf(fp,"\n");
	fclose(fp);
	*/

	return RR_SUCCESS;
}

RRStatus RRBSpline::mfScaleP(
			const RRArray< RRVec2 >& p,
			RRArray< RRVec2 >& sp,
			double& factor)
{	
	sp.setSize(p.getSize());
	factor=0.;
	for(int i=0;i<p.getSize();i++)
		factor=(p[i].getY()>factor)?p[i].getY():factor;
	for(i=0;i<sp.getSize();i++)
		sp[i]=RRVec2(p[i].getX(),p[i].getY()/factor);
	return RR_SUCCESS;
}

RRStatus RRBSpline::mfGaussJordan(
			const int dim,
			double *a,  // a[dim][dim]
			double *b,  // b[dim]
			double *x)  // x[dim]
{
	int i,j,k,pivot;

    int n1=dim+1;
	double *m = new double[dim*(dim+1)];
    for(i=0;i<dim;i++)
    {
		int in,in1;

		in=i*dim;
		in1=i*n1;
		for(j=0;j<dim;j++)
			m[in1+j]=a[in+j];
		m[in1+dim]=b[i];
    }

    for(k=0;k<dim;k++)
    {
		int kn1=k*n1;

		double pvval=0.;
		for(i=k;i<dim;i++)
			if(fabs(m[i*n1+k])>fabs(pvval))
			{
				pivot=i;
				pvval=m[i*n1+k];
			}

		int pn1=pivot*n1;
		for(j=0;j<n1;j++)
		{
			double t=m[pn1+j];
			m[pn1+j]=m[kn1+j];
			m[kn1+j]=t;
		}

		for(j=k;j<n1;j++)
			m[kn1+j]/=pvval;
	
		for(i=0;i<dim;i++)
			if(i!=k)
			{
				int in1=i*n1;
				pvval=m[in1+k];
				for(j=k;j<n1;j++)
				m[in1+j]-=m[kn1+j]*pvval;
			}
    }
	
    for(i=0;i<dim;i++)
		x[i]=m[i*n1+dim];

    delete[] m;
    return RR_SUCCESS;
}

RRStatus RRBSpline::calcCoeff(const RRArray< RRVec2 >& p)
{
	int dim=mKnot.getSize()-1+mDegree;

	double factor;
	RRArray< RRVec2 > sp;
	if(mfScaleP(p,sp,factor)==RR_ERROR)
		return RR_ERROR;

	double *a = new double[dim*dim];
	double *b = new double[dim];
	double *x = new double[dim];

	if(mfMakeMatrix(sp,dim,a,b)==RR_ERROR)
	{
		delete[] a;
		delete[] b;
		delete[] x;
		return RR_ERROR;
	}

	if(mfGaussJordan(dim,a,b,x)==RR_ERROR)
	{
		delete[] a;
		delete[] b;
		delete[] x;
		return RR_ERROR;
	}

	mCoeff.setSize(mKnot.getSize()-1+mDegree);
	for(int i=0;i<mCoeff.getSize();i++)
		mCoeff[i]=x[i]*factor;

	delete[] a;
	delete[] b;
	delete[] x;

	return RR_SUCCESS;
}

double RRBSpline::mfGetKnot(int i)const
{
	i=(i<0)?0:i;
	i=(i>=mKnot.getSize())?mKnot.getSize()-1:i;
	return mKnot[i];
}

double RRBSpline::mfGetCoeff(int i)const
{
	return mCoeff[i+mDegree];
}

int RRBSpline::mfGetPolynomialIndex(double x)const
{
	int j=0;
	while(j<mKnot.getSize() && x>=mfGetKnot(j))
		j++;
	return j-1;
}

double RRBSpline::mfGetBase(int i,int d,double x)const
{
	assert(d>=0);

	if(d==0)
	{
		if(i<0)
			return (x==mfGetKnot(0))?1.:0.;
		else if(i>=mKnot.getSize())
			return (x==mfGetKnot(mKnot.getSize()-1))?1.:0.;
		else
			return (mfGetPolynomialIndex(x)==i)?1.:0.;
	}
	else
	{
		double n0=mfGetBase(i,d-1,x)*(x-mfGetKnot(i));
		n0=(n0==0.)?
			0.:n0/(mfGetKnot(i+d)-mfGetKnot(i));
		double n1=mfGetBase(i+1,d-1,x)*(mfGetKnot(i+d+1)-x);
		n1=(n1==0.)?
			0.:n1/(mfGetKnot(i+d+1)-mfGetKnot(i+1));
		return n0+n1;
	}
}

double RRBSpline::getY(double x)const
{
	if(x==mfGetKnot(0))
		return mfGetCoeff(-mDegree);
	if(x==mfGetKnot(mKnot.getSize()-1))
		return mfGetCoeff(mKnot.getSize()-2);

	int j=mfGetPolynomialIndex(x);
	if(j<0 || j>=mKnot.getSize()) // out of range
		return HUGE_VAL;

	double y=0.;
	for(int i=j-mDegree;i<=j;i++)
		y+=mfGetCoeff(i)*mfGetBase(i,mDegree,x);
	return y;
}

RRBSpline RRBSpline::getDerivative()const
{
	int degree=mDegree-1;
	RRArray< double > coeff(mCoeff.getSize()-1);
	for(int i=0;i<coeff.getSize();i++)
	{
		int j=i-degree;
		coeff[i]=(degree+1)*
			(mfGetCoeff(j)-mfGetCoeff(j-1))/
			(mfGetKnot(j+degree+1)-mfGetKnot(j));
	}

	return RRBSpline(degree,mKnot,coeff);
}

bool operator==(const RRBSpline& b1,const RRBSpline& b2)
{
	return (b1.mDegree==b2.mDegree &&
			b1.mKnot==b2.mKnot &&
			b1.mCoeff==b2.mCoeff)?true:false;
}

bool operator!=(const RRBSpline& b1,const RRBSpline& b2)
{
	return (b1.mDegree!=b2.mDegree ||
			b1.mKnot!=b2.mKnot ||
			b1.mCoeff!=b2.mCoeff)?true:false;
}

//-------------------------------------------------------------------=72
// End of RRBSpline.cpp
//-------------------------------------------------------------------=72

