//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRPixelStatisticsData.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Oct/27/1998:	Copied from eldPixelStatisticsData.cpp
//  Dec/02/1998:	Added member mNumSamples
//
// Bugs:
//
//-------------------------------------------------------------------=72


#include "RRPixelStatisticsData.h"

#include <math.h>

RRPixelStatisticsData::RRPixelStatisticsData()
:mNumSamples(0),
 mMean(0,0,0.),
 mVariance(0,0,0.)
{}

RRStatus RRPixelStatisticsData::mfCheckDataRange(
			const int x,
			const int y)const
{
	int xs,ys;
	mMean.getSize(xs,ys);
	return (x<0 || x>=xs || y<0 || y>=ys)?RR_ERROR:RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::getSize(
			int& xs,
			int& ys)const
{
	mMean.getSize(xs,ys);
	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::setSize(
			const int xs,
			const int ys)
{
	mMean.setSize(xs,ys);
	mVariance.setSize(xs,ys);
	return RR_SUCCESS;
}

double RRPixelStatisticsData::getMeanAt(
			const int x,
			const int y)const
{
	if(mfCheckDataRange(x,y)==RR_ERROR)
		return HUGE_VAL;

	return mMean.getElem(x,y);
}

RRStatus RRPixelStatisticsData::setMeanAt(
			const int x,
			const int y,
			const double mean)
{
	if(mfCheckDataRange(x,y)==RR_ERROR)
		return RR_ERROR;

	mMean.setElem(x,y,mean);
	return RR_SUCCESS;	
}

double RRPixelStatisticsData::getVarianceAt(
			const int x,
			const int y)const
{
	if(mfCheckDataRange(x,y)==RR_ERROR)
		return HUGE_VAL;

	return mVariance.getElem(x,y);
}

RRStatus RRPixelStatisticsData::setVarianceAt(
			const int x,
			const int y,
			const double variance)
{
	if(mfCheckDataRange(x,y)==RR_ERROR)
		return RR_ERROR;

	mVariance.setElem(x,y,variance);
	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::setAllMeanAndVariance(
			const RRField< double >& mean,
			const RRField< double >& variance)
{
	int mxs,mys,vxs,vys;
	mean.getSize(mxs,mys);
	variance.getSize(vxs,vys);
	if(mxs!=vxs || mys!=vys)
		return RR_ERROR;

	mMean=mean;
	mVariance=variance;
	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::save(
			FILE *fp)const
{
	fprintf(fp,"numSamples\t%d\n",mNumSamples);

	int xs,ys;
	mMean.getSize(xs,ys);
	fprintf(fp,"xSize\t%d\tySize\t%d\n",xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double mean=mMean.getElem(x,y);
			double variance=mVariance.getElem(x,y);
			fprintf(fp,"x\t%d\ty\t%d\tmean\t%12le\tvarince\t%12le\n",x,y,mean,variance);
		}

	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::load(
			FILE *fp)
{
	fscanf(fp,"numSamples\t%d\n",&mNumSamples);

	int xs,ys;
	fscanf(fp,"xSize\t%d\tySize\t%d\n",&xs,&ys);
	mMean.setSize(xs,ys);
	mVariance.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int xx,yy;
			double mean,variance;
			fscanf(fp,"x\t%d\ty\t%d\tmean\t%12le\tvarince\t%12le\n",
				&xx,&yy,&mean,&variance);
			mMean.setElem(x,y,mean);
			mVariance.setElem(x,y,variance);
		}

	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::saveBinary(
			FILE *fp)const
{
	fwrite(&mNumSamples,sizeof(int),1,fp);

	int xs,ys;
	mMean.getSize(xs,ys);
	fwrite(&xs,sizeof(int),1,fp);
	fwrite(&ys,sizeof(int),1,fp);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double mean=mMean.getElem(x,y);
			double variance=mVariance.getElem(x,y);
			fwrite(&mean,sizeof(double),1,fp);
			fwrite(&variance,sizeof(double),1,fp);
		}

	return RR_SUCCESS;
}

RRStatus RRPixelStatisticsData::loadBinary(
			FILE *fp)
{
	fread(&mNumSamples,sizeof(int),1,fp);

	int xs,ys;
	fread(&xs,sizeof(int),1,fp);
	fread(&ys,sizeof(int),1,fp);
	mMean.setSize(xs,ys);
	mVariance.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double mean,variance;
			fread(&mean,sizeof(double),1,fp);
			fread(&variance,sizeof(double),1,fp);
			mMean.setElem(x,y,mean);
			mVariance.setElem(x,y,variance);
		}

	return RR_SUCCESS;
}

const RRDataSubstance& RRPixelStatisticsData::operator=(
			const RRDataSubstance& s)
{
	const RRPixelStatisticsData& ss=(const RRPixelStatisticsData&)s;
    return (const RRDataSubstance&)operator=(ss);
}

const RRPixelStatisticsData& RRPixelStatisticsData::operator=(
			const RRPixelStatisticsData& s)
{
	mNumSamples=s.mNumSamples;
	mMean=s.mMean;
	mVariance=s.mVariance;
	return *this;
}

bool RRPixelStatisticsData::operator==(
			const RRPixelStatisticsData& x)const
{
	return (mNumSamples==x.mNumSamples &&
	        mMean==x.mMean && 
			mVariance==x.mVariance)?true:false;
}

//-------------------------------------------------------------------=72
// End of RRPixelStatisticsData.cpp
//-------------------------------------------------------------------=72
