//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RVDocUserDefinePart.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Dec/05/1998:	Created	
//
// Bugs:
//
//-------------------------------------------------------------------=72

#define _RV_DOC_USER_DEFINE_PART_C

#include "stdafx.h"

#include "RVDoc.h"
#include "RVDocUserDefinePart.h"
#include "RVView.h"

#include "RRFileName.h"
#include "RRNumberedFileName.h"
#include "RRPixelStatisticsData.h"
#include "RRImageFileIO.h"

RVDocUserDefinePart::RVDocUserDefinePart()
:mImage(),
 mRadiance(),
 mFocus(NULL),
 mProcessType(RV_EQUALIZING)
{}

RVDocUserDefinePart::~RVDocUserDefinePart()
{}	

void RVDocUserDefinePart::getSize(
			int& xs,
			int& ys,
			int& nchan)const
{
	mRadiance.getSize(xs,ys);
	nchan=3;
}

bool RVDocUserDefinePart::onOpenDocument(
			LPCTSTR lpszPathName,
			const RVDoc *doc)
{
	//mImage=RRFileName(strrchr(lpszPathName,'\\')+1);
	mImage=RRFileName(lpszPathName);

	char ext[64];
	strcpy(ext,strrchr(&mImage,'.'));
	char body[256];
	memset(body,0,256);
	strncpy(body,&mImage,strlen(&mImage)-5); 
	

	if(strcmp(ext,".bmp")==0 || 
	   strcmp(ext,".ppm")==0 || 
	   strcmp(ext,".ppm16")==0)
	{
		if(RRImageFileIO::load(&mImage,RRIFIO_EXT,RRIFIO_PIXELD,mRadiance)==RR_ERROR)
			return false;
	}
	else // fn == <body>.#.psd,  <ext> == .psd
	{	

		RRNumberedFileName nfn(2,body,ext);
		RRArray< RRField< double > > rad(3);
		for(int ch=0;ch<3;ch++)
		{
			RRFileName fch(nfn.getFileName(ch)(0));
			FILE *fp = fopen(fch(0),"rb");
			RRPixelStatisticsData psd;
			psd.loadBinary(fp);
			rad[ch]=psd.getAllMean();
			fclose(fp);
		}

		int xs,ys;
		rad[0].getSize(xs,ys);
		mRadiance.setSize(xs,ys);
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double r=rad[0].getElem(x,y);
				double g=rad[1].getElem(x,y);
				double b=rad[2].getElem(x,y);
				mRadiance.setElem(x,y,RRPixelD(r,g,b,1.));
			}
	}
	
	POSITION pos=doc->GetFirstViewPosition();
	if(pos!=NULL)
	{
		RVView *view=(RVView*)(doc->GetNextView(pos));
		view->mREH.removeFocusRect();
		view->mREH.addFocusRect();
		view->mUDP.setBitmap();
		glDisable(GL_SCISSOR_TEST);
		view->mUDP.onDraw();
	}

	return true;
}

bool RVDocUserDefinePart::mfGetBitmap(
			const RRImage< RRPixelB >& img,
			const int xo,
			const int yo,
			const int xs,
			const int ys,
			const int nchan,
			unsigned char *bitmap)const
{
	int rxs,rys;
	img.getSize(rxs,rys);
	
	if(rxs==0 || rys==0)
		return false; 

	int i=0;
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			unsigned char v[4];
			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
				v[0]=v[1]=v[2]=v[3]=0;
			else
			{
				RRPixelB p(img.getElem(rx,ry));
				v[0]=p.getR();			
				v[1]=p.getG();			
				v[2]=p.getB();			
				v[3]=p.getA();			
			}

			switch(nchan)
			{
			  case 4:
				bitmap[i]=v[0];
				bitmap[i+1]=v[1];
				bitmap[i+2]=v[2];
				bitmap[i+3]=v[3];
				i+=4;
				break;
			  case 3:
				bitmap[i]=v[0];
				bitmap[i+1]=v[1];
				bitmap[i+2]=v[2];
				i+=3;
				break;
			  case 1:
				bitmap[i]=(v[0]+v[1]+v[2])/3;
				i++;
				break;
			}
		}	
	
	return true;
}

bool RVDocUserDefinePart::getBitmap(
			const int xo,
			const int yo,
			const int xs,
			const int ys,
			const int nchan,
			unsigned char *bitmap)const
{
	int rxs,rys;
	mRadiance.getSize(rxs,rys);
	
	if(rxs==0 || rys==0)
		return false; 

	int i=0;
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			unsigned char v[3];
			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
				v[0]=v[1]=v[2]=0;
			else
			{
				RRPixelB p(mRadiance.getElem(rx,ry));
				v[0]=p.getR();			
				v[1]=p.getG();			
				v[2]=p.getB();			
			}

			switch(nchan)
			{
			  case 3:
				bitmap[i]=v[0];
				bitmap[i+1]=v[1];
				bitmap[i+2]=v[2];
				i+=3;
				break;
			  case 1:
				bitmap[i]=(v[0]+v[1]+v[2])/3;
				i++;
				break;
			}
		}	
	
	return true;
}

double *dataR;
int *idxR;

int cmpR(const void *d1,const void *d2)
{
	int *idx1=(int*)d1;
	int *idx2=(int*)d2;

	if(dataR[*idx1]<dataR[*idx2])
		return -1;
	else if(dataR[*idx1]>dataR[*idx2])
		return 1;
	else
		return 0;
}

bool RVDocUserDefinePart::getLocalRadianceBitmap(
			const int xo,
			const int yo,
			const int xs,
			const int ys,
			const int nchan,
			unsigned char *bitmap)const
{
	switch(mProcessType)
	{
	  case RV_EQUALIZING:
		return mfGetLocalRadianceBitmapByEqualizing(xo,yo,xs,ys,nchan,bitmap);
		break;
	  case RV_SCALING:
		return mfGetLocalRadianceBitmapByScaling(xo,yo,xs,ys,nchan,bitmap);
		break;
	  default:
		return false;
	}
}

bool RVDocUserDefinePart::mfGetLocalRadianceBitmapByEqualizing(
			const int xo,
			const int yo,
			const int xs,
			const int ys,
			const int nchan,
			unsigned char *bitmap)const
{
	int rxs,rys;
	mRadiance.getSize(rxs,rys);
	
	if(rxs==0 || rys==0)
		return false; 

	int numdata=xs*ys;
	dataR = new double[numdata];
	idxR = new int[numdata];
	
	int cnt=0;
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
				continue;

			RRPixelD v=mRadiance.getElem(rx,ry);
			dataR[cnt]=(v.getR()+v.getG()+v.getB());

			cnt++;
		}

	int *idx = new int[cnt];
	for(int m=0;m<cnt;m++)
		idx[m]=m;
	qsort(idx,cnt,sizeof(int),cmpR);
	for(m=0;m<cnt;m++)
		idxR[idx[m]]=m;
	delete[] idx;

	int i=0;
	int ccnt=0;
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			unsigned char v[3];
			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
			{
				v[0]=v[1]=v[2]=0;
			}
			else
			{
				RRPixelD p=mRadiance.getElem(rx,ry);
				double lum=(p.getR()+p.getG()+p.getB())/3.;
				double ratio=(double(idxR[ccnt])/cnt)/lum;
				double R=p.getR()*ratio;
				double G=p.getG()*ratio;
				double B=p.getB()*ratio;
				R=(R<0)?0:((R>1)?1:R);
				G=(G<0)?0:((G>1)?1:G);
				B=(B<0)?0:((B>1)?1:B);
				v[0]=unsigned char(R*255+0.5);
				v[1]=unsigned char(G*255+0.5);
				v[2]=unsigned char(B*255+0.5);

				ccnt++;
			}

			switch(nchan)
			{
			  case 3:
				bitmap[i]=v[0];
				bitmap[i+1]=v[1];
				bitmap[i+2]=v[2];
				i+=3;
				break;
			  case 1:
				bitmap[i]=(v[0]+v[1]+v[2])/3;
				i++;
				break;
			}
		}

	delete[] dataR;
	delete[] idxR;
	return true;
}

bool RVDocUserDefinePart::mfGetLocalRadianceBitmapByScaling(
			const int xo,
			const int yo,
			const int xs,
			const int ys,
			const int nchan,
			unsigned char *bitmap)const
{
	const double expand = 4.;

	int rxs,rys;
	mRadiance.getSize(rxs,rys);
	
	if(rxs==0 || rys==0)
		return false; 

	//double maxrad=0.;
	//double minrad=1.;
	double median=0.;
	int cnt=0;
	//for(int y=ys/2-1;y<ys/2+1;y++)
	//	for(int x=xs/2-1;x<xs/2+1;x++)
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
				continue;

			RRPixelD v=mRadiance.getElem(rx,ry);
			median+=v.getR()+v.getG()+v.getB();
			cnt++;

			//double lum=(v.getR()+v.getG()+v.getB())/3.;
			//maxrad=(lum>maxrad)?lum:maxrad;
			//minrad=(lum<minrad)?lum:minrad;
		}
	median/=cnt*3;

	int i=0;
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int rx=xo+x;
			int ry=yo+y;

			unsigned char v[3];
			if(rx<0 || rx>=rxs || ry<0 || ry>=rys)
				v[0]=v[1]=v[2]=0;
			else
			{
				RRPixelD p=mRadiance.getElem(rx,ry);
				double lum=(p.getR()+p.getG()+p.getB())/3.;

				//double ratio=(lum-minrad)/(maxrad-minrad)/lum;				
				double ratio=((lum-median)*expand+0.5)/lum;

				double R=p.getR()*ratio;
				double G=p.getG()*ratio;
				double B=p.getB()*ratio;
				R=(R<0)?0:((R>1)?1:R);
				G=(G<0)?0:((G>1)?1:G);
				B=(B<0)?0:((B>1)?1:B);

				v[0]=unsigned char(R*255+0.5);
				v[1]=unsigned char(G*255+0.5);
				v[2]=unsigned char(B*255+0.5);
			}

			switch(nchan)
			{
			  case 3:
				bitmap[i]=v[0];
				bitmap[i+1]=v[1];
				bitmap[i+2]=v[2];
				i+=3;
				break;
			  case 1:
				bitmap[i]=(v[0]+v[1]+v[2])/3;
				i++;
				break;
			}
		}	
	
	return true;
}

RRStatus RVDocUserDefinePart::addFocusRect(
			RVFocusRect *fr)
{
	mFocus=fr;
	fr->setDoc(this);
	return RR_SUCCESS;
}

RRStatus RVDocUserDefinePart::removeFocusRect(
			RVFocusRect *fr)
{
	mFocus=NULL;
	return RR_SUCCESS;
}

void RVDocUserDefinePart::onFocusSize(
			const int size,
			const RVDoc *doc)
{
	if(size<0)
		mFocus->setWidth(300);
	else
		mFocus->setWidth(size);

	POSITION pos=doc->GetFirstViewPosition();
	if(pos!=NULL)
	{
		RVView *view=(RVView*)(doc->GetNextView(pos));
		glDisable(GL_SCISSOR_TEST);
		view->mUDP.onDraw();
	}
}

//-------------------------------------------------------------------=72
// End of RVDocUserDefinePart.cpp
//-------------------------------------------------------------------=72
