//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRSelectProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Oct/27/1998:	Copied from eldSelectProc.cpp
//
// Bugs:
//
//-------------------------------------------------------------------=72

//#define RRSP_DEBUG

#include "RRSelectProc.h"

#include <math.h>

const double alpha = 0.05; // 5% as rejection ratio

const int RRSP_CHI_FREEDOM[8] = {
			0,		// kernel size = 0
			0,		//			   1
			3,		//			   2
			8,		//			   3
			15,		//			   4
			24,		//			   5
			35,		//			   6
			48};	//			   7

const double RRSP_UPPER_CHI_SQUARE[8] = { // chi^2(freedom,alpha)
			0.000000,	// chi^2(0,0.05)
			0.000000,	// chi^2(0,0.05)
			7.814728,	// chi^2(3,0.05)
			15.507313,	// chi^2(8,0.05)
			24.995790,	// chi^2(15,0.05)
			36.415029,	// chi^2(24,0.05)
			49.801850,	// chi^2(35,0.05)
			65.170769};	// chi^2(48,0.05)

#ifdef RRSP_DEBUG
# include "RRBMPImage.h"
# include "RRPixelB.h"
# include "RRPixelD.h"
const char *RRSP_DEBUG_IMAGE_0 = "RRSP.debug.0.bmp";
const char *RRSP_DEBUG_IMAGE_1 = "RRSP.debug.1.bmp";
#endif // RRSP_DEBUG

#ifdef RRSP_DEBUG
void RRSP_WriteSelectedPixelsTo(
			const RRField< double >& mean,
			const RRField< int >& select,
			RRImage< RRPixelB >& img)
{
	int xs,ys;
	mean.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double v=(select.getElem(x,y)==0)?1.:mean.getElem(x,y);
			img.setElem(x,y,RRPixelB(RRPixelD(v,v,v,1.)));
		}
}
#endif // RRSP_DEBUG

RRSelectProc::RRSelectProc(
			const RRArray< RRData* >& in,	
									// in0 -> RRPixelStatisticsData
									// in1 -> RRPixelStatisticsData
			RRData *out)			// RR2DDoubleArrayData
:RRProcess("RRSelectProc",in,out)
{}

RRStatus RRSelectProc::mfSelectByTrimming(
			RRField< int >& select)
{
	int xs,ys;
	select.getSize(xs,ys);

	int trimX= int(xs*(1.-mTrimRatio))/2;
	int trimY= int(ys*(1.-mTrimRatio))/2;

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			s=(x<trimX || x>xs-trimX ||
			   y<trimY || y>ys-trimY)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectProc::mfSelectBySampling(
			RRField< int >& select)
{
	int xs,ys;
	select.getSize(xs,ys);

	int trimX= int(xs*(1.-mTrimRatio))/2;
	int trimY= int(ys*(1.-mTrimRatio))/2;
	int rectX=xs-trimX*2;
	int rectY=ys-trimY*2;
	
	int intvl=(mMaxSamplingNum<0)?1:int(sqrt(rectX*rectY/mMaxSamplingNum));

	int offsetX=(rectX-int(rectX/intvl)*intvl)/2;
	int offsetY=(rectY-int(rectY/intvl)*intvl)/2;

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			s=((x-trimX-offsetX)%intvl!=0 ||
			   (y-trimY-offsetY)%intvl!=0)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectProc::mfSelectByLimitting(
			const RRField< double >& mean,
			RRField< int >& select)
{
	int xs,ys;
	select.getSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			double v=mean.getElem(x,y);
			s=(v>mSaturationLevel || v<mNoiseLevel)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectProc::mfSelectByNoiseVarianceTest(
			const RRField< double >& variance,
			const RRField< double >& noise,
			RRField< int >& select)
{
	const double upperX2 = RRSP_UPPER_CHI_SQUARE[mKernelSize];
	const int numsamples = mKernelSize*mKernelSize;

	const double noisevar=mNoiseLevel*mNoiseLevel;

	int xs,ys;
	select.getSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			double v=variance.getElem(x,y);
			double n=noise.getElem(x,y);
			//n=(n<=0)?1.e-12:n;
			n=(n<=0)?noisevar:n;

			double test=numsamples*v/n;
			s=(test>upperX2)?0:s;
			//s=(n>significantNoise)?0:s;
			s=(n>noisevar)?0:s;
		
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectProc::mfSelectPixels(
			const RRField< double >& mean,
			const RRField< double >& variance,
			const RRField< double >& noise,
			RRField< int >& select)
{
	int xs,ys;
	mean.getSize(xs,ys);
	select=RRField< int >(xs,ys,1);

	if(mfSelectByTrimming(select)==RR_ERROR)
		return RR_ERROR;

//	if(mfSelectBySampling(select)==RR_ERROR)
//		return RR_ERROR;

	if(mfSelectByLimitting(mean,select)==RR_ERROR)
		return RR_ERROR;

	if(mfSelectByNoiseVarianceTest(variance,noise,select)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

double *cmpx0,*cmpx1;

int RRSP_Cmp(
			const void *p1,
			const void *p2)
{
	double x1=cmpx1[*((int*)p1)];
	double x2=cmpx1[*((int*)p2)];
	if(x1>x2)
		return 1;
	else if(x1<x2)
		return -1;
	else
		return 0;
}

RRStatus RRSelectProc::mfMakeImArray(
			const RRField< double >& mean0,
			const RRField< double >& mean1,
			const RRField< int >& select0,
			const RRField< int >& select1,
			RRArray< RRArray< double > >& im)
{
	int xs,ys;
	mean0.getSize(xs,ys);

	cmpx0 = new double[xs*ys];
	cmpx1 = new double[xs*ys];

	int cnt=0;
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select0.getElem(x,y)*select1.getElem(x,y);
			if(s==1)
			{
				cmpx0[cnt]=mean0.getElem(x,y);
				cmpx1[cnt]=mean1.getElem(x,y);
				//im[0][cnt]=mean0.getElem(x,y);
				//im[1][cnt]=mean1.getElem(x,y);
				cnt++;
			}
		}

	if(cnt==0)
		return RR_ERROR;

	int *idx = new int[cnt];
	for(int i=0;i<cnt;i++)
		idx[i]=i;
	qsort(idx,cnt,sizeof(int),RRSP_Cmp);

	im.setSize(2);
	int numdata=(cnt<mMaxSamplingNum || mMaxSamplingNum<0)?cnt:mMaxSamplingNum;
	im[0].setSize(numdata);
	im[1].setSize(numdata);

	{
	/* Do not use because of  lack of uniformity of sampling 
		im[0][0]=cmpx0[idx[0]];
		im[1][0]=cmpx1[idx[0]];
		for(i=1;i<numdata;i++)
		{
			int id=int(double(cnt)/numdata*i);
			im[0][i]=cmpx0[idx[id]];
			im[1][i]=cmpx1[idx[id]];
		}
	*/
	 // corrected sampling process
		int n=0;
		im[0][n]=cmpx0[idx[0]];
		im[1][n]=cmpx1[idx[0]];
		double pn=im[1][n];
		for(int d=1;d<cnt;d++)
		{
			int id=idx[d];
			double pd=cmpx1[id];
			if(fabs(pn-pd)<1./numdata)
			{
			}
			else
			{
				n++;
				im[0][n]=cmpx0[id];
				im[1][n]=cmpx1[id];
				pn=pd;
			}
		}
		n++;
		im[0].setSize(n);
		im[1].setSize(n);
	}


	delete[] idx;
	delete[] cmpx0;
	delete[] cmpx1;

	return RR_SUCCESS;
}

RRStatus RRSelectProc::doIt()
{
	if(mInput.getSize()<5)
		return RR_ERROR;

	RRSelectParamData& param=(RRSelectParamData&)mInput[4]->getSubstance();

	mTrimRatio=param.getTrimRatio();
	mSaturationLevel=param.getSaturationLevel();
	mNoiseLevel=param.getNoiseLevel();
	mMaxSamplingNum=param.getMaxSamplingNumber();
	mKernelSize=param.getKernelSize();

	RRPixelStatisticsData& t0=(RRPixelStatisticsData&)mInput[0]->getSubstance();
	RRPixelStatisticsData& t1=(RRPixelStatisticsData&)mInput[1]->getSubstance();
	RRPixelStatisticsData& s0=(RRPixelStatisticsData&)mInput[2]->getSubstance();
	RRPixelStatisticsData& s1=(RRPixelStatisticsData&)mInput[3]->getSubstance();

	const RRField< double >& mean0=s0.getAllMean();
	const RRField< double >& mean1=s1.getAllMean();
	const RRField< double >& variance0=s0.getAllVariance();
	const RRField< double >& variance1=s1.getAllVariance();
	const RRField< double >& noise0=t0.getAllVariance();
	const RRField< double >& noise1=t1.getAllVariance();

	RRField< int > select0;
	RRField< int > select1;

	if(mfSelectPixels(mean0,variance0,noise0,select0)==RR_ERROR)
		return RR_ERROR;

#ifdef RRSP_DEBUG
	RRWOBMPImage img0(RRSP_DEBUG_IMAGE_0);
	RRSP_WriteSelectedPixelsTo(mean0,select0,img0);
#endif // RRSP_DEBUG

	if(mfSelectPixels(mean1,variance1,noise1,select1)==RR_ERROR)
		return RR_ERROR;
	
#ifdef RRSP_DEBUG
	RRWOBMPImage img1(RRSP_DEBUG_IMAGE_1);
	RRSP_WriteSelectedPixelsTo(mean1,select1,img1);
#endif // RRSP_DEBUG

	RRArray< RRArray< double > > im(2);
	if(mfMakeImArray(mean0,mean1,select0,select1,im)==RR_ERROR)
		return RR_ERROR;

	RR2DDoubleArrayData& out=(RR2DDoubleArrayData&)mOutput->getSubstance();
	out.setData(im);

	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRSelectProc.cpp
//-------------------------------------------------------------------=72
