//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRSelectAutoTrimProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Apr/02/1999:	Arranged from eldSelectLowVarianceAutoTrim.C
//
// Bugs:
//
//-------------------------------------------------------------------=72

//#define RRSATP_DEBUG

#include "RRSelectAutoTrimProc.h"
#include "RRBSpline.h"
#include "RRVec2.h"

#include <math.h>

const double alpha = 0.05; // 5% as rejection ratio

const int RRATSP_CHI_FREEDOM[8] = {
			0,		// kernel size = 0
			0,		//			   1
			3,		//			   2
			8,		//			   3
			15,		//			   4
			24,		//			   5
			35,		//			   6
			48};	//			   7

const double RRATSP_UPPER_CHI_SQUARE[8] = { // chi^2(freedom,alpha)
			0.000000,	// chi^2(0,0.05)
			0.000000,	// chi^2(0,0.05)
			7.814728,	// chi^2(3,0.05)
			15.507313,	// chi^2(8,0.05)
			24.995790,	// chi^2(15,0.05)
			36.415029,	// chi^2(24,0.05)
			49.801850,	// chi^2(35,0.05)
			65.170769};	// chi^2(48,0.05)


#ifdef RRSATP_DEBUG
# include "RRBMPImage.h"
# include "RRPixelB.h"
# include "RRPixelD.h"
const char *RRSATP_DEBUG_IMAGE_0 = "RRSATP.DEBUG.0.bmp";
const char *RRSATP_DEBUG_IMAGE_1 = "RRSATP.DEBUG.1.bmp";
#endif // RRSATP_DEBUG

#ifdef RRSATP_DEBUG
void RRATSP_WriteSelectedPixelsTo(
			const RRField< double >& mean,
			const RRField< int >& select,
			RRImage< RRPixelB >& img)
{
	int xs,ys;
	mean.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double v=(select.getElem(x,y)==0)?1.:mean.getElem(x,y);
			img.setElem(x,y,RRPixelB(RRPixelD(v,v,v,1.)));
		}
}
#endif // RRSATP_DEBUG

RRSelectAutoTrimProc::RRSelectAutoTrimProc(
			const RRArray< RRData* >& in,	
									// in0 -> RRPixelStatisticsData
									// in1 -> RRPixelStatisticsData
			RRData *out)			// RR2DDoubleArrayData
:RRProcess("RRSelectAutoTrimProc",in,out)
{}

RRStatus RRSelectAutoTrimProc::mfSelectByTrimming(
			RRField< int >& select)
{// round triming
	int xs,ys;
	select.getSize(xs,ys);

	int centerX= xs/2;
	int centerY= ys/2;
	double radius = sqrt(centerX*centerX+centerY*centerY);
	double trimradius=radius*mTrimRatio;

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

 			double rx=x-centerX;
			double ry=y-centerY;
			s=(sqrt(rx*rx+ry*ry)>trimradius)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::mfSelectBySampling(
			RRField< int >& select)
{
	int xs,ys;
	select.getSize(xs,ys);

	int trimX= int(xs*(1.-mTrimRatio))/2;
	int trimY= int(ys*(1.-mTrimRatio))/2;
	int rectX=xs-trimX*2;
	int rectY=ys-trimY*2;
	
	int intvl=(mMaxSamplingNum<0)?1:int(sqrt(rectX*rectY/mMaxSamplingNum));

	int offsetX=(rectX-int(rectX/intvl)*intvl)/2;
	int offsetY=(rectY-int(rectY/intvl)*intvl)/2;

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			s=((x-trimX-offsetX)%intvl!=0 ||
			   (y-trimY-offsetY)%intvl!=0)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::mfSelectByLimitting(
			const RRField< double >& mean,
			RRField< int >& select)
{
	int xs,ys;
	select.getSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			double v=mean.getElem(x,y);
			s=(v>mSaturationLevel || v<mNoiseLevel)?0:s;
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::mfSelectByNoiseVarianceTest(
			const RRField< double >& variance,
			const RRField< double >& noise,
			RRField< int >& select)
{
	const double upperX2 = RRATSP_UPPER_CHI_SQUARE[mKernelSize];
	const int numsamples = mKernelSize*mKernelSize;

	const double noisevar=mNoiseLevel*mNoiseLevel;

	int xs,ys;
	select.getSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select.getElem(x,y);
			if(s==0)
				continue;

			double v=variance.getElem(x,y);
			double n=noise.getElem(x,y);
			//n=(n<=0)?1.e-12:n;
			n=(n<=0)?noisevar:n;

			double test=numsamples*v/n;
			s=(test>upperX2)?0:s;
			//s=(n>significantNoise)?0:s;
			s=(n>noisevar)?0:s;
		
			select.setElem(x,y,s);
		}

	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::mfSelectPixels(
			const RRField< double >& mean,
			const RRField< double >& variance,
			const RRField< double >& noise,
			RRField< int >& select)
{
	int xs,ys;
	mean.getSize(xs,ys);
	select=RRField< int >(xs,ys,1);

	if(mfSelectByTrimming(select)==RR_ERROR)
		return RR_ERROR;

//	if(mfSelectBySampling(select)==RR_ERROR)
//		return RR_ERROR;

	if(mfSelectByLimitting(mean,select)==RR_ERROR)
		return RR_ERROR;

	if(mfSelectByNoiseVarianceTest(variance,noise,select)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

double *RRSATP_cmpx0,*RRSATP_cmpx1;

int RRSATP_Cmp(
			const void *p1,
			const void *p2)
{
	double x1=RRSATP_cmpx1[*((int*)p1)];
	double x2=RRSATP_cmpx1[*((int*)p2)];
	if(x1>x2)
		return 1;
	else if(x1<x2)
		return -1;
	else
		return 0;
}

RRStatus RRSelectAutoTrimProc::mfMakeImArray(
			const RRField< double >& mean0,
			const RRField< double >& mean1,
			const RRField< int >& select0,
			const RRField< int >& select1,
			RRArray< RRArray< double > >& im)
{
	int xs,ys;
	mean0.getSize(xs,ys);

	RRSATP_cmpx0 = new double[xs*ys];
	RRSATP_cmpx1 = new double[xs*ys];

	int cnt=0;
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			int s=select0.getElem(x,y)*select1.getElem(x,y);
			if(s==1)
			{
				RRSATP_cmpx0[cnt]=mean0.getElem(x,y);
				RRSATP_cmpx1[cnt]=mean1.getElem(x,y);
				cnt++;
			}
		}

	if(cnt==0)
		return RR_ERROR;

	int *idx = new int[cnt];
	for(int i=0;i<cnt;i++)
		idx[i]=i;
	qsort(idx,cnt,sizeof(int),RRSATP_Cmp);

	im.setSize(2);
	int numdata=(cnt<mMaxSamplingNum || mMaxSamplingNum<0)?cnt:mMaxSamplingNum;
	im[0].setSize(numdata);
	im[1].setSize(numdata);

	{
		int n=0;
		im[0][n]=RRSATP_cmpx0[idx[0]];
		im[1][n]=RRSATP_cmpx1[idx[0]];
		double pn=im[1][n];
		for(int d=1;d<cnt;d++)
		{
			int id=idx[d];
			double pd=RRSATP_cmpx1[id];
			if(fabs(pn-pd)<1./numdata)
			{
			}
			else
			{
				n++;
				im[0][n]=RRSATP_cmpx0[id];
				im[1][n]=RRSATP_cmpx1[id];
				pn=pd;
			}
		}
		n++;
		im[0].setSize(n);
		im[1].setSize(n);
	}


	delete[] idx;
	delete[] RRSATP_cmpx0;
	delete[] RRSATP_cmpx1;

	return RR_SUCCESS;
}

int RRSATP_cmp(
			const void *p1,
			const void *p2)
{
	double x1= *(double*)p1;
	double x2= *(double*)p2;
	if(x1>x2)
		return 1;
	else if(x1<x2)
		return -1;
	else
		return 0;
}

RRStatus RRSelectAutoTrimProc::mfCalcKnots(
			const RRArray< double >& x,
			RRArray< double >& knot)
{
	const int numKnots = knot.getSize();

	double itvl=1./(numKnots-1);
	for(int k=0;k<numKnots;k++)
		knot[k]=(k==numKnots-1)?1.:k*itvl;

	double *xary = new double[x.getSize()];
	for(int i=0;i<x.getSize();i++)
		xary[i]=x[i];

	qsort(xary,x.getSize(),sizeof(double),RRSATP_cmp);

	int imidx=x.getSize()-1,nkidx=1;
	RRArray< double > nknot(numKnots);
	nknot[0]=1;
	for(k=numKnots-2;k>0;k--)
	{
		int cnt=0;
		while(imidx>=0 && xary[imidx]>knot[k])
		{
			imidx--;
			cnt++;
		}

		if(cnt!=0)
		{
			nknot[nkidx]=knot[k];
			nkidx++;
		}
	}
	nknot[nkidx]=0;
	nkidx++;
	nknot.setSize(nkidx);

	knot.setSize(nkidx);
	knot[0]=0;
	imidx=0;
	nkidx=1;
	for(k=nknot.getSize()-2;k>0;k--)
	{
		int cnt=0;
		while(imidx<x.getSize() && xary[imidx]<nknot[k])
		{
			imidx++;
			cnt++;
		}

		if(cnt!=0)
		{
			knot[nkidx]=nknot[k];
			nkidx++;
		}
	}
	knot[nkidx]=1;
	nkidx++;
	knot.setSize(nkidx);
	
	delete[] xary;
	return RR_SUCCESS;	
}

RRStatus RRSelectAutoTrimProc::mfMakeSpline(
			const RRArray< double >& x,
			const RRArray< double >& y,
			RRBSpline& spl)
{
	const int numKnots = 21;
	RRArray< double > knot(numKnots);	
	if(mfCalcKnots(x,knot)==RR_ERROR)
		return RR_ERROR;

	spl.setKnot(knot);

	RRArray< RRVec2 > p(x.getSize());
	for(int i=0;i<p.getSize();i++)
		p[i]=RRVec2(x[i],y[i]);

	if(spl.calcCoeff(p)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::mfIsImScattered(
			const RRArray< RRArray< double > >& im,
			bool& ret)
{
	const int degree = 1;
	RRBSpline spline0(degree);
	if(mfMakeSpline(im[0],im[1],spline0)==RR_ERROR)
		return RR_ERROR;

	RRBSpline spline1(degree);
	if(mfMakeSpline(im[1],im[0],spline1)==RR_ERROR)
		return RR_ERROR;

	/*
	double maxdis=0;
	for(int i=0;i<im[0].getSize();i++)
	{
		double dis0=fabs(im[0][i]-spline1.getY(im[1][i]));
		double dis1=fabs(im[1][i]-spline0.getY(im[0][i]));
	
		double sl0=spline0.getDerivative().getY(im[0][i]);
		double sl1=spline1.getDerivative().getY(im[1][i]);

		//double dis=(sl0<sl1)?dis1:dis0;
		double dis=dis0*dis1/sqrt(dis0*dis0+dis1*dis1);
		if(dis>maxdis)
			maxdis=dis;
	}
	*/
	double var0=0;
	double var1=0;
	for(int i=0;i<im[0].getSize();i++)
	{
		double dis0=im[0][i]-spline1.getY(im[1][i]);
		double dis1=im[1][i]-spline0.getY(im[0][i]);
		var0+=dis0*dis0;
		var1+=dis1*dis1;
	}
	var0/=im[0].getSize();
	var1/=im[1].getSize();
	double varn = mNoiseLevel*mNoiseLevel;

	ret=(var0<varn && var1<varn)?false:true;
	//ret=(maxdis<mNoiseLevel*2)?false:true;
	return RR_SUCCESS;
}

RRStatus RRSelectAutoTrimProc::doIt()
{
	if(mInput.getSize()<5)
		return RR_ERROR;

	RRSelectParamData& param=(RRSelectParamData&)mInput[4]->getSubstance();

	mTrimRatio=1.; 
	mSaturationLevel=param.getSaturationLevel();
	mNoiseLevel=param.getNoiseLevel();
	mMaxSamplingNum=param.getMaxSamplingNumber();
	mKernelSize=param.getKernelSize();

	RRPixelStatisticsData& t0=(RRPixelStatisticsData&)mInput[0]->getSubstance();
	RRPixelStatisticsData& t1=(RRPixelStatisticsData&)mInput[1]->getSubstance();
	RRPixelStatisticsData& s0=(RRPixelStatisticsData&)mInput[2]->getSubstance();
	RRPixelStatisticsData& s1=(RRPixelStatisticsData&)mInput[3]->getSubstance();

	const RRField< double >& mean0=s0.getAllMean();
	const RRField< double >& mean1=s1.getAllMean();
	const RRField< double >& variance0=s0.getAllVariance();
	const RRField< double >& variance1=s1.getAllVariance();
	const RRField< double >& noise0=t0.getAllVariance();
	const RRField< double >& noise1=t1.getAllVariance();

	RRField< int > select0;
	RRField< int > select1;
	RRArray< RRArray< double > > im(2);

	mTrimRatio=1.;
	int cnt=0;
	bool end=false;
	while(!end)
	{
		if(mfSelectPixels(mean0,variance0,noise0,select0)==RR_ERROR)
			return RR_ERROR;

//#ifdef RRSATP_DEBUG
//		RRWOBMPImage img0(RRSATP_DEBUG_IMAGE_0);
//		RRSP_WriteSelectedPixelsTo(mean0,select0,img0);
//#endif // RRSATP_DEBUG

		if(mfSelectPixels(mean1,variance1,noise1,select1)==RR_ERROR)
			return RR_ERROR;
	
//#ifdef RRSATP_DEBUG
//		RRWOBMPImage img1(RRSATP_DEBUG_IMAGE_1);
//		RRSP_WriteSelectedPixelsTo(mean1,select1,img1);
//#endif // RRSATP_DEBUG

		if(mfMakeImArray(mean0,mean1,select0,select1,im)==RR_ERROR)
			return RR_ERROR;

		bool scattered;
		if(mfIsImScattered(im,scattered)==RR_ERROR)
			return RR_ERROR;

		if(scattered==false || mTrimRatio<0.2)
		{
#ifdef RRSATP_DEBUG
			RRWOBMPImage img0(RRSATP_DEBUG_IMAGE_0);
			RRATSP_WriteSelectedPixelsTo(mean0,select0,img0);
			RRWOBMPImage img1(RRSATP_DEBUG_IMAGE_1);
			RRATSP_WriteSelectedPixelsTo(mean1,select1,img1);
#endif // RRSATP_DEBUG

			end=true;			
		}
		else
		{
			//printf("RRSelectAutoTrimProc::doIt: auto trim loop r=%lf\n",mTrimRatio);
			mTrimRatio-=0.1;
		}

		cnt++;
	}

	RR2DDoubleArrayData& out=(RR2DDoubleArrayData&)mOutput->getSubstance();
	out.setData(im);

	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRSelectAutoTrimProc.cpp
//-------------------------------------------------------------------=72
