//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRCombineProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Nov/21/1998:	Copied from eldRadianceMap.cpp
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "RRCombineProc.h"

//debug
#include "RRBMPImage.h"
#include "RRPixelD.h"
#include "RRNumberedFileName.h"
//debug

#include <math.h>

const double RRCBP_GETX_PRECISION = 1.e-6;
const double RRCBP_SATURATION_LEVEL = 0.990;
const double RRCBP_NOISE_LEVEL = 0.010;

RRCombineProc::RRCombineProc(
			RRArray< RRData* >& in,	
								// in[0]: RRIaCurveData
								// in[1]...in[n-2]: RRDoubleFieldData[n-2]
				                // in[n-1]: RRCombineParamData
			RRData *out)	    // RRPixelStatisticsData
:RRProcess("RRCombineProc",in,out)
{}

double RRCombineProc::mfCalcWeight(
			const double im)
{
	RRIaCurveData& icd=(RRIaCurveData&)(mInput[0]->getSubstance());
	RRPolynomial ia=icd.getIaFunc();
	RRPolynomial dia=ia.getDerivative(1);

	if(im<RRCBP_NOISE_LEVEL)
		return -HUGE_VAL;
	else if(im>RRCBP_SATURATION_LEVEL)
		return HUGE_VAL;
	else
	{
		return ia.getY(im)/dia.getY(im);		
	}

}

double RRCombineProc::mfCalcBestIm()
{
	const double precision = 0.001;
	double maxw=0,maxim=0;
	for(double im=RRCBP_NOISE_LEVEL;im<=RRCBP_SATURATION_LEVEL;im+=precision)
	{
		double w=mfCalcWeight(im);
		if(w>maxw) 
		{
			maxw=w;
			maxim=im;
		}
	}
	
	return maxim;	
}

RRStatus RRCombineProc::doIt()
{
	const int numim=mInput.getSize()-2;

	RRIaCurveData& icd=(RRIaCurveData&)(mInput[0]->getSubstance());
	RRPolynomial iafunc=icd.getIaFunc();
	RRArray< double > ratio=icd.getExposureRatio();

	const int numR=ratio.getSize();
	double sum=0.;
	for(int i=0;i<numR+1;i++)
	{
		double rr=1.;
		for(int j=i;j<numR;j++)
			rr*=ratio[j];
		sum+=rr;
	}	

	RRArray< double > aper(numR+1);
	aper[aper.getSize()-1]=(numR+1)/sum;
	for(int r=aper.getSize()-1;r>0;r--)
		aper[r-1]=aper[r]*ratio[r-1];

	const double gim = mfCalcBestIm();
	const double gia = iafunc.getY(gim);
	printf("bestIa = %le\n",gia);

	int xs,ys;
	//RRPixelStatisticsData& psd=(RRPixelStatisticsData&)(mInput[1]->getSubstance());
	//psd.getAllMean().getSize(xs,ys);
	RRDoubleFieldData& psd=(RRDoubleFieldData&)(mInput[1]->getSubstance());
	psd.getData().getSize(xs,ys);

	RRField< double > radm(xs,ys);
	//RRField< double > radv(xs,ys);

	double maxrad=0.;

#define TEST_ORG
#define TEST_TWOPASS // Need also to define TEST_ORG

	
//debug
	RRArray< RRImage< RRPixelB > > iaimg(numim);
	for(int h=0;h<numim;h++)
		iaimg[h].setSize(xs,ys);
//debug

 	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRArray< double > wght(numim);
			RRArray< double > ia(numim);
			RRArray< double > im(numim);

			for(int n=0;n<numim;n++)
			{
				//RRPixelStatisticsData& psd=(RRPixelStatisticsData&)(mInput[n+1]->getSubstance());
				RRDoubleFieldData& psd=(RRDoubleFieldData&)(mInput[n+1]->getSubstance());
				im[n]=psd.getData().getElem(x,y);
				ia[n]=iafunc.getY(im[n]);
				wght[n]=mfCalcWeight(im[n]);
			}

			double siam=0.;
			double siav=0.;

			double sw=0.;
			int maxwght=0;
			int whtcnt=0,blkcnt=0;
			for(int i=0;i<numim;i++)
			{
				double iam=ia[i]/aper[i];

				//debug
				iaimg[i].setElem(x,y,RRPixelB(RRPixelD(iam,iam,iam,1)));
				//debug

				if(wght[i]==HUGE_VAL)
				{
					whtcnt=numim-i;
					break;
				}
				else if(wght[i]== -HUGE_VAL)
				{
					blkcnt=i+1;
					siam=0.;
					siav=0.;
					sw=0.;
				}
				else
				{
#ifdef TEST_ORG
					// original 
					siam+=wght[i]*iam;
					siav+=wght[i]*iam*iam;
					sw+=wght[i];
#endif // TEST_ORG										
				}
			}

			if(whtcnt==numim)
			{
				siam=HUGE_VAL;
				siav=0.;
			}
			else if(blkcnt==numim)
			{
				siam= -HUGE_VAL;
				siav=0.;
			}
			else
			{
				siam/=sw;
				siav=siav/sw-siam*siam;

#ifdef TEST_TWOPASS

				if(siam<RRCBP_NOISE_LEVEL || siam>RRCBP_SATURATION_LEVEL)
				{
				
				// test 10/24
				RRArray< double > ria(numim);
				for(int j=0;j<numim;j++)
					ria[j]=siam*aper[j];
				double sia=0.,dia=HUGE_VAL;
				int sj;
				for(j=0;j<numim;j++)
				{
					if(wght[j]!=HUGE_VAL && wght[j]!= -HUGE_VAL)
						if(fabs(ria[j]-gia)<dia)
						{
							dia=fabs(ria[j]-gia);
							sia=ria[j];
							sj=j;
						}
				}
				siam=ia[sj]/aper[sj];
				//siam=im[sj]; // just test
				}

#endif // TEST_TWOPASS

				maxrad=(siam>maxrad)?siam:maxrad;
			}

			radm.setElem(x,y,siam);
			//radv.setElem(x,y,siav);
		}

	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double m=radm.getElem(x,y);
			m=(m== -HUGE_VAL)?0:((m==HUGE_VAL)?maxrad:m);
			radm.setElem(x,y,m/maxrad);
			//radv.setElem(x,y,radv.getElem(x,y)/(maxrad*maxrad));
		}

	//RRPixelStatisticsData& psdrad=(RRPixelStatisticsData&)(mOutput->getSubstance());
	//psdrad.setAllMeanAndVariance(radm,radv);
	RRDoubleFieldData& psdrad=(RRDoubleFieldData&)(mOutput->getSubstance());
	psdrad.setData(radm);

//debug
	RRNumberedFileName iaimgfile(numim-1,"debug.",".bmp");
	for(h=0;h<numim;h++)
	{
		RRWOBMPImage iaimgf(&(iaimgfile.getFileName(h)));
		iaimgf=iaimg[h];
	}
//debug

	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRCombineProc.cpp
//-------------------------------------------------------------------=72
