//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRCombineProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Nov/21/1998:	Copied from eldRadianceMap.cpp
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "RRCombineProc.h"

//#define RRCMP_DEBUG

#ifdef RRCMP_DEBUG
# include "RRBMPImage.h"
# include "RRPixelD.h"
# include "RRNumberedFileName.h"
#endif // RRCMP_DEBUG

#include <math.h>

const double RRCBP_GETX_PRECISION = 1.e-6;
const double RRCBP_SATURATION_LEVEL = 0.990;
const double RRCBP_NOISE_LEVEL = 0.010;
const double sdThreshold = 0.02;

RRCombineProc::RRCombineProc(
			RRArray< RRData* >& in,	
								// in[0]: RRIaCurveData
								// in[1]...in[n-2]: RRDoubleFieldData[n-2]
				                // in[n-1]: RRCombineParamData
			RRData *out)	    // RRPixelStatisticsData
:RRProcess("RRCombineProc",in,out)
{}

double RRCombineProc::mfCalcWeight(
			const double im)
{
	RRIaCurveData& icd=(RRIaCurveData&)(mInput[0]->getSubstance());
	RRPolynomial ia=icd.getIaFunc();
	RRPolynomial dia=ia.getDerivative(1);

	if(im<RRCBP_NOISE_LEVEL)
		return -HUGE_VAL;
	else if(im>RRCBP_SATURATION_LEVEL)
		return HUGE_VAL;
	else
	{
		return ia.getY(im)/dia.getY(im);		
	}

}

double RRCombineProc::mfCalcBestIm()
{
	const double precision = 0.001;
	double maxw=0,maxim=0;
	for(double im=RRCBP_NOISE_LEVEL;im<=RRCBP_SATURATION_LEVEL;im+=precision)
	{
		double w=mfCalcWeight(im);
		if(w>maxw) 
		{
			maxw=w;
			maxim=im;
		}
	}
	
	return maxim;	
}

RRStatus RRCombineProc::mfCalcExposure(
			const RRArray< double >& ratio,
			RRArray< double >& exposure)
{
	if(ratio.getSize()<1)
		return RR_ERROR;

	const int numR = ratio.getSize();
	const int numE = numR+1;

	exposure.setSize(numE);

	double sum=0.;
	for(int i=0;i<numE;i++)
	{
		double rr=1.;
		for(int j=i;j<numR;j++)
			rr*=ratio[j];
		sum+=rr;
	}	

	exposure[numE-1]=numE/sum;
	for(int r=numE-1;r>0;r--)
		exposure[r-1]=exposure[r]*ratio[r-1];

	return RR_SUCCESS;
}

RRStatus RRCombineProc::doIt()
{
	const int numIm = mInput.getSize()-2;
	assert(numIm>0);

	RRIaCurveData& icd=(RRIaCurveData&)(mInput[0]->getSubstance());
	RRPolynomial iafunc=icd.getIaFunc();
	RRArray< double > ratio=icd.getExposureRatio();

	RRArray< double > expo;
	if(mfCalcExposure(ratio,expo)==RR_ERROR)
		return RR_ERROR;

	const double gim = mfCalcBestIm();
	const double gia = iafunc.getY(gim);
	printf("bestIa = %le at %le\n",gia,gim);

	int xs,ys;
	RRDoubleFieldData& psd=(RRDoubleFieldData&)(mInput[1]->getSubstance());
	psd.getData().getSize(xs,ys);
	assert(xs>0 && ys>0);

	RRField< double > rad(xs,ys);

	double maxrad=0.;

#ifdef RRCMP_DEBUG
	RRArray< RRImage< RRPixelB > > iaimg(numIm);
	for(int h=0;h<numIm;h++)
		iaimg[h].setSize(xs,ys);
#endif // RRCMP_DEBUG

 	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
		
			RRArray< double > wght(numIm);
			RRArray< double > ia(numIm);

			double sia=0.,sw=0.;
			int whtcnt=0,blkcnt=0;
			int maxwghtidx=0;
			double mn=0.,vr=0.;
			int nmn=0;
			for(int n=0;n<numIm;n++)
			{
				RRDoubleFieldData& psd=(RRDoubleFieldData&)(mInput[n+1]->getSubstance());
				wght[n]=mfCalcWeight(psd.getData().getElem(x,y));
				ia[n]=iafunc.getY(psd.getData().getElem(x,y))/expo[n];
				ia[n]=(ia[n]<0.)?0.:ia[n];

#ifdef RRCMP_DEBUG
				iaimg[n].setElem(x,y,RRPixelB(RRPixelD(ia[n],ia[n],ia[n],1)));
#endif // RRCMP_DEBUG

				if(wght[n]==HUGE_VAL)
				{
					whtcnt=numIm-n;
					break;
				}
				else if(wght[n]== -HUGE_VAL)
				{
					blkcnt=n+1;
					sia=0.;
					sw=0.;
				}
				else
				{
					sia+=wght[n]*ia[n];
					sw+=wght[n];

					if(wght[n]>wght[maxwghtidx])
						maxwghtidx=n;

					mn+=ia[n];
					vr+=ia[n]*ia[n];
					nmn++;
				}
			}

			mn/=nmn;
			vr=vr/nmn-mn*mn;

			if(whtcnt==numIm)
			{
				sia=HUGE_VAL;
			}
			else if(blkcnt==numIm)
			{
				sia= -HUGE_VAL;
			}
			else 
			{
				if(sqrt(vr)<sdThreshold)
				{
					sia=ia[maxwghtidx]*wght[maxwghtidx];
					sw=wght[maxwghtidx];
				}

				sia/=sw;
				maxrad=(sia>maxrad)?sia:maxrad;
			}

			rad.setElem(x,y,sia);
		}

	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double m=rad.getElem(x,y);
			m=(m== -HUGE_VAL)?0:((m==HUGE_VAL)?maxrad:m);
			rad.setElem(x,y,m/maxrad);
		}

	RRDoubleFieldData& psdrad=(RRDoubleFieldData&)(mOutput->getSubstance());
	psdrad.setData(rad);

#ifdef RRCMP_DEBUG
	RRNumberedFileName iaimgfile(numIm-1,"debug.",".bmp");
	for(h=0;h<numIm;h++)
	{
		RRWOBMPImage iaimgf(&(iaimgfile.getFileName(h)));
		iaimgf=iaimg[h];
	}
#endif // RRCMP_DEBUG

	return RR_SUCCESS;
}



//-------------------------------------------------------------------=72
// End of RRCombineProc.cpp
//-------------------------------------------------------------------=72
