//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRCombineProc.h
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Nov/21/1998:	Copied from eldRadianceMap.h
//	Dec/15/1998:	Improved algorithms of combining and color balancing
//
// Bugs:
//
//-------------------------------------------------------------------=72
//
// RRCombineProc combine input images into a high dynamic range image. 
//
// in[0]:			RRFileListData
//					File list of image files.
// in[1]:			RRFileListData
//					File list of response curve data files 
//					(RRIaCurveData). 
// in[2]:			RRCombineParamData
//					parameters
// out:				RRFileListData
//					File list of combined image. 
//
//	A high dynamic rage image is obtained by the following steps.
//	
//	For each pixel,
//	(1) pixel values from input images are calibrated by the given 
//		response curve.
//	(2) Those calibrated pixel values are scaled by scale factors which 
//		are computed from given exposure ratios.
//	(3) One pixel value is computed from the scaled pixel values by the 
//		one method in the followings.
//		- 	All scaled pixel values are weighted by data reliability and
//			averaged. 
//		-	The most reliability pixel value is selected.
//	(4)	The result is set to the output high dyamic range image. 
//	
//	RRCombineProc process three channel (RGB) image. 
//
//	Parameters for RRCombineProc are defined in RRCombineParamData.
//	See RRCombineParamData.h for detail. 
//
//-------------------------------------------------------------------=72

#ifndef _RR_COMBINE_PROC_H
#define _RR_COMBINE_PROC_H

#include "RRProcess.h"
#include "RRCombineParamData.h"
#include "RRFileListData.h"
#include "RRDoubleFieldData.h"
#include "RRIaCurveData.h"

class RRCombineProc : public RRProcess
{
  private:  

	double mSaturationLevel;
	double mNoiseLevel;
	double mCheckIVarianceCriterion;

	RRPolynomial mCurve[3];
	RRPolynomial mDeriv[3];
	RRArray< double > mExposure[3];

  private:

	RRStatus mfSetParam();
	RRStatus mfSetCurvesAndExposures();
	RRStatus mfCalcExposure(
			const RRArray< double >& ratio,
			RRArray< double >& exposure);
	RRStatus mfLoadImagesToField(
			const int ch,
			const RRFileListData& flist,
			RRArray< RRField< double > >& fld);

	double mfCalcWeight(
			const int ch,
			const double im);
	double mfCalcBestM(
			const int ch);
	RRStatus mfCombine(
			const int ch,
			const RRArray< RRField< double > >& img,
			RRField< double >& rad);
	double mfCalcI(
			const int ch,
			const RRArray< RRField< double > >& img,
			const int x,
			const int y);

  public:    
  
  	RRCombineProc(
			RRArray< RRData* >& in,
								// in[0]: RRFileListData (images)
								// in[1]: RRFileListData (RRIaCurveData)
				                // in[2]: RRCombineParamData
			RRData *out);	    // RRFileListData (RRDoubleFieldData)

	RRStatus doIt();
	// execute operation
};

#endif // _RR_COMBINE_PROC_H

//-------------------------------------------------------------------=72
// End of RRCombineProc.h
//-------------------------------------------------------------------=72