//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRCombineProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Nov/21/1998:	Copied from eldRadianceMap.cpp
//	Dec/15/1998:	Improved algorithms of combining and color balancing
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "RRCombineProc.h"
#include "RRNumberedFileName.h"
#include "RRImageFileIO.h"
#include "RRFiledData.h"

//#define RRCMP_DEBUG

#include <math.h>

const double RRCBP_GETX_PRECISION = 1.e-6;
const double RRCBP_BESTIM_PRECISION = 0.001;
const int numCh = 3;

RRCombineProc::RRCombineProc(
			RRArray< RRData* >& in,	
								// in[0]: RRFileListData
								// in[1]: RRFileListData
				                // in[2]: RRCombineParamData
			RRData *out)	    // RRPFileListData
:RRProcess("RRCombineProc",in,out)
{}

RRStatus RRCombineProc::mfSetParam()
{
	const RRCombineParamData& param = (const RRCombineParamData&)mInput[2]->getSubstance();
	mSaturationLevel=param.getSaturationLevel();
	mNoiseLevel=param.getNoiseLevel();
	mCheckIVarianceCriterion=param.getCheckIVarianceCriterion();
	return RR_SUCCESS;
}

RRStatus RRCombineProc::mfSetCurvesAndExposures()
{
	const RRFileListData& flist = (const RRFileListData&)mInput[1]->getSubstance();
	if(flist.getNumFiles()!=numCh)
		return RR_ERROR;
		
	for(int c=0;c<numCh;c++)
	{
		RRROFiledData< RRIaCurveData > cdata(flist.getFileName(c));
		const RRIaCurveData& icd = (const RRIaCurveData&)cdata.getSubstance();
		mCurve[c]=icd.getIaFunc();
		mDeriv[c]=mCurve[c].getDerivative(1);
		if(mfCalcExposure(icd.getExposureRatio(),mExposure[c])==RR_ERROR)
			return RR_ERROR;
	}	

	return RR_SUCCESS;
}

RRStatus RRCombineProc::mfCalcExposure(
			const RRArray< double >& ratio,
			RRArray< double >& exposure)
{
	if(ratio.getSize()<1)
		return RR_ERROR;

	const int numR = ratio.getSize();
	const int numE = numR+1;

	exposure.setSize(numE);

	double sum=0.;
	for(int i=0;i<numE;i++)
	{
		double rr=1.;
		for(int j=i;j<numR;j++)
			rr*=ratio[j];
		sum+=rr;
	}	

	exposure[numE-1]=numE/sum;
	for(int r=numE-1;r>0;r--)
		exposure[r-1]=exposure[r]*ratio[r-1];

	return RR_SUCCESS;
}

double RRCombineProc::mfCalcWeight(
			const int ch,
			const double im)
{
	if(im<mNoiseLevel)
		return -HUGE_VAL;
	else if(im>mSaturationLevel)
		return HUGE_VAL;
	else
	{
		double w=mCurve[ch].getY(im)/mDeriv[ch].getY(im);
		return (w<0.)?0.:w;		
	}
}

double RRCombineProc::mfCalcBestM(
			const int ch)
{
	double maxw=0,maxm=0;
	for(double m=mNoiseLevel;m<=mSaturationLevel;m+=RRCBP_BESTIM_PRECISION)
	{
		double w=mfCalcWeight(ch,m);
		if(w>maxw) 
		{
			maxw=w;
			maxm=m;
		}
	}
	
	return maxm;	
}

RRStatus RRCombineProc::mfLoadImagesToField(
			const int ch,
			const RRFileListData& flist,
			RRArray< RRField< double > >& fld)
{
	const int numFiles = flist.getNumFiles();
	fld.setSize(numFiles);

	for(int i=0;i<numFiles;i++)
	{
		RRIFIODataType channel;
		switch(ch)
		{
		  case 0:
			channel=RRIFIO_DOUBLE_R; // load R channel
			break;
		  case 1:
			channel=RRIFIO_DOUBLE_G; // load G channel
			break;
		  case 2:
			channel=RRIFIO_DOUBLE_B; // load B channel
			break;
		}
		
		// Do loading
		if(RRImageFileIO::load(flist.getFileName(i),
		                       RRIFIO_EXT,
							   channel,
							   fld[i])==RR_ERROR)
			return RR_ERROR;

	}	
	return RR_SUCCESS;
}

RRStatus RRCombineProc::mfCombine(
			const int ch,
			const RRArray< RRField< double > >& img,
			RRField< double >& rad)
{
	const double bestM = mfCalcBestM(ch);
	const double bestI = mCurve[ch].getY(bestM);
	printf("\tbestIa = %le at %le\n",bestI,bestM);

	int xs,ys;
	img[0].getSize(xs,ys);
	if(xs==0 || ys==0)
		return RR_ERROR;

	rad.setSize(xs,ys);
	double maxrad=0.;
	double minrad=HUGE_VAL;
 	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double radval=mfCalcI(ch,img,x,y);
			rad.setElem(x,y,radval);
			
			if(radval!= -HUGE_VAL && radval!=HUGE_VAL)
			{
				maxrad=(radval>maxrad)?radval:maxrad;
				minrad=(radval<minrad)?radval:minrad;
			}
		}

	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double m=rad.getElem(x,y);
			m=(m== -HUGE_VAL)?minrad:((m==HUGE_VAL)?maxrad:m);
			rad.setElem(x,y,m/maxrad);
		}

	return RR_SUCCESS;
}		
		
double RRCombineProc::mfCalcI(
			const int ch,
			const RRArray< RRField< double > >& img,
			const int x,
			const int y)
{
	const int numIm = img.getSize();
	
	RRArray< double > wght(numIm);
	RRArray< double > ia(numIm);

	double sia=0.,sw=0.;
	int whtcnt=0,blkcnt=0;
	int maxwghtidx=0;
	double maxwght= -HUGE_VAL;
	double mn=0.,vr=0.;
	int nmn=0;
	for(int n=0;n<numIm;n++)
	{
		ia[n]=mCurve[ch].getY(img[n].getElem(x,y))/mExposure[ch][n];
		ia[n]=(ia[n]<0.)?0.:ia[n];
		wght[n]=mfCalcWeight(ch,img[n].getElem(x,y));

		if(wght[n]==HUGE_VAL)
		{
			whtcnt=numIm-n;
			break;
		}
		else if(wght[n]== -HUGE_VAL)
		{
			blkcnt=n+1;
			sia=0.;
			sw=0.;

			maxwghtidx=0;
			maxwght= -HUGE_VAL;

			mn=0.;
			vr=0.;
			nmn=0;
		}
		else
		{
			sia+=wght[n]*ia[n];
			sw+=wght[n];
			
			if(wght[n]>maxwght)
			{
				maxwghtidx=n;
				maxwght=wght[n];
			}

			mn+=ia[n];
			vr+=ia[n]*ia[n];
			nmn++;
		}
	}

	nmn=(nmn==0)?1:nmn;
	mn/=nmn;
	vr=vr/nmn-mn*mn;

	if(whtcnt==numIm)
		sia=HUGE_VAL;
	else if(blkcnt==numIm)
		sia= -HUGE_VAL;
	else if(vr<mCheckIVarianceCriterion)
		sia= ia[maxwghtidx];
	else
		sia/=sw;

	return sia;
}

RRStatus RRCombineProc::doIt()
{
	const RRFileListData& imglist = (const RRFileListData&)mInput[0]->getSubstance();
	const int numIm = imglist.getNumFiles();
	if(numIm<=0)
		return RR_ERROR;

	if(mfSetParam()==RR_ERROR)
		return RR_ERROR;
		
	if(mfSetCurvesAndExposures()==RR_ERROR)
		return RR_ERROR;
		
	RRArray< RRField< double > > rad(numCh);
	for(int c=0;c<numCh;c++)
	{
		printf("RRCombineProc: processing channel %d\n",c);

		RRArray< RRField< double > > img;
		if(mfLoadImagesToField(c,imglist,img)==RR_ERROR)
			return RR_ERROR;

		if(mfCombine(c,img,rad[c])==RR_ERROR)
			return RR_ERROR;
	}
	
#ifdef RRCMP_DEBUG
	RRImageFileIO::save(rad[0],RRIFIO_DOUBLE_R,RRIFIO_EXT,"RRCMP.debug.ppm16");
	RRImageFileIO::save(rad[1],RRIFIO_DOUBLE_G,RRIFIO_EXT,"RRCMP.debug.ppm16");
	RRImageFileIO::save(rad[2],RRIFIO_DOUBLE_B,RRIFIO_EXT,"RRCMP.debug.ppm16");
#endif // RRCMP_DEBUG

	RRFileListData& outfile=(RRFileListData&)(mOutput->getSubstance());
	for(c=0;c<numCh;c++)
	{
		RRFileName fname=outfile.getFileName(c);
		RRDoubleFieldData fd;
		if(fd.setData(rad[c])==RR_ERROR)
			return RR_ERROR;
		FILE *fp=fopen(&fname,"wb");
		if(fd.saveBinary(fp)==RR_ERROR)
		{
			fclose(fp);
			return RR_ERROR;
		}
		fclose(fp);
	}
	
	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRCombineProc.cpp
//-------------------------------------------------------------------=72