//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRColorBalanceProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//	Dec/16/1998:	Created
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "RRColorBalanceProc.h"
#include "RRNumberedFileName.h"
#include "RRImageFileIO.h"

#include <math.h>

const int RRCBP_WARNING_DATA_COUNT = 100;
const int numCh = 3;

RRColorBalanceProc::RRColorBalanceProc(
			RRArray< RRData* >& in,	
								// in[0]: RRFileListData
								// in[1]: RRFileListData
				                // in[2]: RRCombineParamData
			RRData *out)	    // RRPFileListData
:RRProcess("RRColorBalanceProc",in,out)
{}

RRStatus RRColorBalanceProc::mfSetParam()
{
	const RRCombineParamData& param = (const RRCombineParamData&)mInput[2]->getSubstance();
	mSaturationLevel=param.getSaturationLevel();
	mNoiseLevel=param.getNoiseLevel();
	mTrimRatio=param.getTrimRatio();
	mBestMLuminance=param.getBestMLuminance();
	mSelectNeutralColorCriterion=param.getSelectNeutralColorCriterion();
	return RR_SUCCESS;
}

RRStatus RRColorBalanceProc::mfMakeMData(
			RRField< RRPixelD >& mimg)
{
	printf("RRColorBalanceProc::mfMakeMData\n");

	const RRFileListData& imd = (const RRFileListData&)mInput[0]->getSubstance();
	const int numFiles = imd.getNumFiles();
	if(numFiles<1)
		return RR_ERROR;

	int xs,ys;
	{
		RRField< RRPixelD > img;
		if(RRImageFileIO::load(imd.getFileName(0),
		                       RRIFIO_EXT,
							   RRIFIO_PIXELD,
							   img)==RR_ERROR)
			return RR_ERROR;
		img.getSize(xs,ys);
		mimg=img;
	}

	for(int i=1;i<numFiles;i++)
	{
		RRField< RRPixelD > img;
		if(RRImageFileIO::load(imd.getFileName(i),
		                       RRIFIO_EXT,
							   RRIFIO_PIXELD,
							   img)==RR_ERROR)
			return RR_ERROR;

		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				RRPixelD p=img.getElem(x,y);
				double lump=(p.getR()+p.getG()+p.getB())/3.;
				RRPixelD q=mimg.getElem(x,y);
				double lumq=(q.getR()+q.getG()+q.getB())/3.;
				
				if(p.getR()>mNoiseLevel && p.getR()<mSaturationLevel &&
				   p.getG()>mNoiseLevel && p.getG()<mSaturationLevel &&
				   p.getB()>mNoiseLevel && p.getB()<mSaturationLevel &&
				   fabs(lump-mBestMLuminance)<fabs(lumq-mBestMLuminance))
					mimg.setElem(x,y,p);
			}
	}

	return RR_SUCCESS;
} 

RRStatus RRColorBalanceProc::mfMakeIData(
			RRField< RRPixelD >& iimg)
{
	printf("RRColorBalanceProc::mfMakeIData\n");

	const RRFileListData& flist = (const RRFileListData&)mInput[1]->getSubstance();
	const int numFiles = flist.getNumFiles();
	if(numFiles!=numCh)
		return RR_ERROR;

	int xs,ys;
	{
		FILE *fp=fopen(flist.getFileName(0),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		fld.getSize(xs,ys);
		iimg.setSize(xs,ys);
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setR(v);
				iimg.setElem(x,y,p);
			}
	}

	{
		FILE *fp=fopen(flist.getFileName(1),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setG(v);
				iimg.setElem(x,y,p);
			}
	}

	{
		FILE *fp=fopen(flist.getFileName(2),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setB(v);
				iimg.setElem(x,y,p);
			}
	}

	return RR_SUCCESS;
} 

RRStatus RRColorBalanceProc::mfCalcScales(
			const RRField< RRPixelD >& mimg,
			const RRField< RRPixelD >& iimg,
			RRArray< double >& k) //k[3]
{
	printf("RRColorBalanceProc::mfCalcScales\n");

	double IrgMrg(0),IrbMrb(0),IggMrr(0),IggMbb(0),IbbMrr(0),IbbMgg(0),IgbMgb(0);

	int xs,ys;
	iimg.getSize(xs,ys);

	const int left = int(xs-xs*mTrimRatio)/2; 
	const int bottom = int(ys-ys*mTrimRatio)/2; 
	const int right = xs-left; 
	const int top = ys-bottom; 

	int cnt=0;
	for(int y=bottom;y<top;y++)
		for(int x=left;x<right;x++)
		{
			RRPixelD ia=iimg.getElem(x,y);
			RRPixelD im=mimg.getElem(x,y);

			// non-neutral colors are rejected.
			double mm=(im.getR()+im.getG()+im.getB())/3.;
			if(fabs(im.getR()-mm)/mm>mSelectNeutralColorCriterion ||
			   fabs(im.getG()-mm)/mm>mSelectNeutralColorCriterion ||
			   fabs(im.getB()-mm)/mm>mSelectNeutralColorCriterion)
				continue;

			IrgMrg+=ia.getR()*ia.getG()*im.getR()*im.getG();
			IrbMrb+=ia.getR()*ia.getB()*im.getR()*im.getB();
			IggMrr+=ia.getG()*ia.getG()*im.getR()*im.getR();
			IggMbb+=ia.getG()*ia.getG()*im.getB()*im.getB();
			IbbMrr+=ia.getB()*ia.getB()*im.getR()*im.getR();
			IbbMgg+=ia.getB()*ia.getB()*im.getG()*im.getG();
			IgbMgb+=ia.getG()*ia.getB()*im.getG()*im.getB();
			cnt++;
		}

	printf("\tdata count = %d\n",cnt);
	if(cnt<RRCBP_WARNING_DATA_COUNT)
	{
		printf("\twarning:\ttoo few data.\n\tCheck mSelectNeutralColorCriterion param.\n");
	}	

	double denom=(IggMrr+IggMbb)*(IbbMrr+IbbMgg)-IgbMgb*IgbMgb;
	double fracG=IrgMrg*(IbbMrr+IbbMgg)+IrbMrb*IgbMgb;
	double fracB=IrbMrb*(IggMrr+IggMbb)+IrgMrg*IgbMgb;

	k[0]=1.;
	k[1]=fracG/denom;
	k[2]=fracB/denom;

	double kmax=0;
	for(int i=0;i<3;i++)
		kmax=(k[i]>kmax)?k[i]:kmax;

	for(i=0;i<3;i++)
		k[i]/=kmax;
	printf("\tk = (%le\t%le\t%le)\n",k[0],k[1],k[2]);

	return RR_SUCCESS;
}

RRStatus RRColorBalanceProc::mfScaleIData(
			const RRField< RRPixelD >& iimg,
			RRArray< double >& k, //k[3]
			RRField< RRPixelD >& cbal)
{
	printf("RRColorBalanceProc::mfScaleIData\n");

	int xs,ys;
	iimg.getSize(xs,ys);
	cbal.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=iimg.getElem(x,y);
			RRPixelD q(p.getR()*k[0],p.getG()*k[1],p.getB()*k[2],1.);
			cbal.setElem(x,y,q);
		}

	return RR_SUCCESS;
}

RRStatus RRColorBalanceProc::doIt()
{
	if(mfSetParam()==RR_ERROR)
		return RR_ERROR;

	RRField< RRPixelD > mimg;
	if(mfMakeMData(mimg)==RR_ERROR)
		return RR_ERROR;

	RRField< RRPixelD > iimg;
	if(mfMakeIData(iimg)==RR_ERROR)
		return RR_ERROR;

	RRArray< double > k(3);
	if(mfCalcScales(mimg,iimg,k)==RR_ERROR)
		return RR_ERROR;

	RRField< RRPixelD > cbal;
	if(mfScaleIData(iimg,k,cbal)==RR_ERROR)
		return RR_ERROR;

	RRFileListData& outfile=(RRFileListData&)(mOutput->getSubstance());
	RRFileName fname=outfile.getFileName(0);
	if(RRImageFileIO::save(cbal,RRIFIO_PIXELD,RRIFIO_EXT,&fname)==RR_ERROR)
		return RR_ERROR;
	
	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRColorBalanceProc.cpp
//-------------------------------------------------------------------=72