//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRCalibrateProc.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Date:                 Oct/31/1998
// Version:              1.0
//
// Modification History:
//  Oct/31/1998:	Copied from eldIaCurveFromN.cpp
//
// Bugs:
//
// Classes:
//
// Notes:
//
//-------------------------------------------------------------------=72

//#define RRCLP_DEBUG

#include "RRCalibrateProc.h"

#include <math.h>

const double RRCP_GETX_PRECISION = 1.e-6;
int RRCalibrateProc::debugFileCount = 0;

#ifdef RRCLP_DEBUG
const char *debugFile = "RRCLP.debug.";
#endif // RRCLP_DEBUG

RRCalibrateProc::RRCalibrateProc(
			RRArray< RRData* >& in,	
								// in[0]...in[n-2]: RR2DDoubleArrayData[n]
				                // in[n-1]: RRCalibrateParamData
			RRData *out)	    // RRIaCurveData
:RRProcess("RRCalibrateProc",in,out)
{}

double RRCalibrateProc::mfCalcBase(
			const int idx,
			const double im0,
			const double im1,
			const double cR)
{
	return pow(im0,idx)-cR*pow(im1,idx);
}

RRStatus RRCalibrateProc::mfGaussJordan(
			const int dim,
			double *a,  // a[dim][dim]
			double *b,  // b[dim]
			double *x)  // x[dim]
{
	int i,j,k,pivot;

    int n1=dim+1;
	double *m = new double[dim*(dim+1)];
    for(i=0;i<dim;i++)
    {
		int in,in1;

		in=i*dim;
		in1=i*n1;
		for(j=0;j<dim;j++)
			m[in1+j]=a[in+j];
		m[in1+dim]=b[i];
    }

    for(k=0;k<dim;k++)
    {
		int kn1=k*n1;

		double pvval=0.;
		for(i=k;i<dim;i++)
			if(fabs(m[i*n1+k])>fabs(pvval))
			{
				pivot=i;
				pvval=m[i*n1+k];
			}

		int pn1=pivot*n1;
		for(j=0;j<n1;j++)
		{
			double t=m[pn1+j];
			m[pn1+j]=m[kn1+j];
			m[kn1+j]=t;
		}

		for(j=k;j<n1;j++)
			m[kn1+j]/=pvval;
	
		for(i=0;i<dim;i++)
			if(i!=k)
			{
				int in1=i*n1;
				pvval=m[in1+k];
				for(j=k;j<n1;j++)
				m[in1+j]-=m[kn1+j]*pvval;
			}
    }
	
    for(i=0;i<dim;i++)
		x[i]=m[i*n1+dim];

    delete[] m;
    return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfMakeMatrix(
			const RRArray< RRArray< RRArray< double > > >& im,
	        const int dim,
			const RRArray< double >& cR,
			double *a, // a[dim][dim]
			double *b) // b[dim]
{
	int numR=cR.getSize();

	int idxN=dim;
	for(int i=0;i<dim;i++)
	{
		int idxI=i;

		for(int j=0;j<dim;j++)
		{
			int idxJ=j;

			double s=0.;
			for(int r=0;r<numR;r++)
			{
				int numdata=im[r][0].getSize();

				for(int n=0;n<numdata;n++)
				{
					double bi=mfCalcBase(idxI,im[r][0][n],im[r][1][n],cR[r]);
					double bj=mfCalcBase(idxJ,im[r][0][n],im[r][1][n],cR[r]);
					double bn=mfCalcBase(idxN,im[r][0][n],im[r][1][n],cR[r]);
					s+=(bi-bn)*(bj-bn);
				}
			}
			a[i*dim+j]=s;
		}
	
		double z=0.;
		for(int r=0;r<numR;r++)
		{
			int numdata=im[r][0].getSize();
			for(int n=0;n<numdata;n++)
			{
				double bi=mfCalcBase(idxI,im[r][0][n],im[r][1][n],cR[r]);
				double bn=mfCalcBase(idxN,im[r][0][n],im[r][1][n],cR[r]);
				z+=(bi-bn)*bn;
			}
		}
		b[i]= -z;
	}
	
	return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfUpdateCoeff(
			const RRArray< RRArray< RRArray< double > > >& im,
			const int order,
			const RRArray< double >& cR,
			RRArray< double >& coeff)
{
	int dim=order;

	double *a = new double[dim*dim];
	double *b = new double[dim];
	double *x = new double[dim];

	if(mfMakeMatrix(im,dim,cR,a,b)==RR_ERROR)
	{
		delete[] a;
		delete[] b;
		delete[] x;
		return RR_ERROR;
	}

	if(mfGaussJordan(dim,a,b,x)==RR_ERROR)
	{
		delete[] a;
		delete[] b;
		delete[] x;
		return RR_ERROR;
	}

	coeff[dim]=1.;
	for(int i=0;i<dim;i++)
	{
		coeff[i]=x[i];
		coeff[dim]-=x[i];
	}

	delete[] a;
	delete[] b;
	delete[] x;

	return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfCheckSimpleIncreasing(
			const RRArray< double >& coeff)
{
	RRPolynomial poly=RRPolynomial(coeff);
	const double checkPrecision = 0.01;

	for(double x=0;x<1;x+=checkPrecision)
	{
		double y0=poly.getY(x);
		double y1=poly.getY(x+checkPrecision);
		if(y0>y1)
			return RR_ERROR;
	}
	return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfEvaluateError(
			const RRArray< RRArray< RRArray< double > > >& im,
			const RRArray< double >& cR,
			const RRArray< double >& coeff,
			double& err)
{
	int numR=cR.getSize();
	int snum=0;
	err=0.;
	for(int r=0;r<numR;r++)
	{
		int numdata=im[r][0].getSize();
		for(int i=0;i<numdata;i++)
		{
			double e=0.;
			for(int j=0;j<coeff.getSize();j++)
				e+=coeff[j]*mfCalcBase(j,im[r][0][i],im[r][1][i],cR[r]);
			err+=pow(e,2);
		}
		snum+=numdata;
	}
	err/=snum;
	return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfEstimateIaFunc(
			const int maxOrder,
			const RRArray< RRArray< RRArray< double > > >& im,
			const RRArray< double >& cR,
			RRPolynomial& ia,
			double& err)
{
	double minErr=HUGE_VAL;
	RRArray< double > minCoeff;
	for(int od=1;od<=maxOrder;od++)
	{
		RRArray< double > tCoeff(od+1);
		if(mfUpdateCoeff(im,od,cR,tCoeff)==RR_ERROR)
			return RR_ERROR;

		if(mfCheckSimpleIncreasing(tCoeff)==RR_ERROR)
			continue;

		double tErr;
		if(mfEvaluateError(im,cR,tCoeff,tErr)==RR_ERROR)
				return RR_ERROR;

		if(tErr<minErr)
		{
			minErr=tErr;
			minCoeff=tCoeff;
		}
	}		

	ia=RRPolynomial(minCoeff);
	err=minErr;
	return RR_SUCCESS;
}

RRStatus RRCalibrateProc::mfRenewR(
			const RRArray< RRArray< RRArray< double > > >& im,
			const RRArray< double >& cR,
			const RRPolynomial& ia,
			RRArray< double >& nR)
{
	int numR=cR.getSize();

	for(int r=0;r<numR;r++)
	{
		int numdata=im[r][0].getSize();
		double sia0=0.,sia1=0.;
		for(int n=0;n<numdata;n++)
		{
			double ia0=ia.getY(im[r][0][n]);
			double ia1=ia.getY(im[r][1][n]);
			sia0+=ia0;
			sia1+=ia1;
		}
		nR[r]=sia0/sia1;
	}

	return RR_SUCCESS;
}

bool RRCalibrateProc::mfCheckConvergence(
			const double convergenceLevel,
			const RRPolynomial& pre,
			const RRPolynomial& cur)
{
	const int dim = 100;
	for(int i=0;i<=dim;i++)
	{
		double im=1./dim*i;
		double delta=cur.getY(im)-pre.getY(im);
		if(fabs(delta)>convergenceLevel)
			return false;
	}
	return true;
}

RRStatus RRCalibrateProc::mfOutput(
			const RRArray< double >& cR,
			const RRPolynomial& ia)
{
    RRIaCurveData& out=(RRIaCurveData&)mOutput->getSubstance();
	out.setIaFunc(ia);
	out.setExposureRatio(cR);

/*	
	int numR=cR.getSize();
	RRArray< double > aper(numR+1);

	double sum=0.;
	for(int i=0;i<numR;i++)
	{
		double rr=1.;
		for(int j=0;j<i;j++)
			rr*=cR[j];
		sum+=rr;
	}	

	aper[aper.getSize()-1]=numR/sum;
	for(int r=aper.getSize()-1;r>0;r--)
		aper[r-1]=aper[r]*cR[r-1];

	out.setAperture(aper);
*/
	
	return RR_SUCCESS;

}

RRStatus RRCalibrateProc::doIt()
{
	const RRCalibrateParamData& param=(const RRCalibrateParamData&)mInput[mInput.getSize()-1]->getSubstance();

	const int numR=param.getExposureRatio().getSize();

	RRArray< double > cR(numR);
	for(int i=0;i<numR;i++)
		cR[i]=param.getExposureRatio()[i];

	RRArray< RRArray< RRArray< double > > > im(numR);	
	for(i=0;i<numR;i++)
	{
		const RR2DDoubleArrayData& in=(const RR2DDoubleArrayData&)mInput[i]->getSubstance();
		im[i]=in.getData();
	}

#ifdef RRCLP_DEBUG
	char dfile[256];
	sprintf(dfile,"%s%d",debugFile,debugFileCount);
	debugFileCount++;
	printf("\tdebug file = %s\n",dfile);
	FILE *debugFp = fopen(dfile,"w");
	mfPrintDataNames(debugFp);
#endif // RRCLP_DEBUG

	RRPolynomial ia,preia(0);
	int iter=0;
	while(1)
	{
		double err;
		if(mfEstimateIaFunc(param.getMaxOrder(),im,cR,ia,err)==RR_ERROR)
			return RR_ERROR;

		RRArray< double > nR(numR);			
		if(mfRenewR(im,cR,ia,nR)==RR_ERROR)
			return RR_ERROR;	
	
#ifdef RRCLP_DEBUG
		mfPrintData(debugFp,iter,ia,err,cR,nR);
#endif // RRCLP_DEBUG

		if(mfCheckConvergence(param.getConvergenceLevel(),preia,ia)==true)
			break;
		else
		{
			cR=nR;
			preia=ia;
		}
		iter++;
	}

	if(mfOutput(cR,ia)==RR_ERROR)	
		return RR_ERROR;

#ifdef RRCLP_DEBUG
		fclose(debugFp);
#endif // RRCLP_DEBUG

	return RR_SUCCESS;
}



// debug printing routines

void RRCalibrateProc::mfPrintDataNames(
			FILE *fp)
{
	const RRCalibrateParamData& param=(const RRCalibrateParamData&)mInput[mInput.getSize()-1]->getSubstance();

	int i;

	fprintf(fp,"coeff\t");
	for(i=0;i<=param.getMaxOrder();i++)
		fprintf(fp,"eC%d\t",i);
	fprintf(fp,"\t");

	fprintf(fp,"err&R\t");
	fprintf(fp,"error\t");
	for(i=0;i<param.getExposureRatio().getSize();i++)
		fprintf(fp,"cR%d\t",i);
	for(i=0;i<param.getExposureRatio().getSize();i++)
		fprintf(fp,"nR%d\t",i);
	fprintf(fp,"\t");

	fprintf(fp,"Ia\t");
	for(i=0;i<=100;i++)
		fprintf(fp,"%lf\t",double(i)/100);
	fprintf(fp,"\t");

	fprintf(fp,"\n");
	fflush(fp);
}

void RRCalibrateProc::mfPrintData(
			FILE *fp,
			const int iter,
	        const RRPolynomial& ia,
			const double err,
			const RRArray< double >& cR,
			const RRArray< double >& nR)
{
	const RRCalibrateParamData& param=(const RRCalibrateParamData&)mInput[mInput.getSize()-1]->getSubstance();

	int i;
	
	fprintf(fp,"%d\t",iter);
	const RRArray< double >& coeff=ia.getCoeff();
	for(i=0;i<=param.getMaxOrder();i++)
		fprintf(fp,"%le\t",(i<coeff.getSize())?coeff[i]:0.);
	fprintf(fp,"\t");

	fprintf(fp,"%d\t",iter);
	fprintf(fp,"%le\t",err);
	for(i=0;i<param.getExposureRatio().getSize();i++)
		fprintf(fp,"%le\t",cR[i]);
	for(i=0;i<param.getExposureRatio().getSize();i++)
		fprintf(fp,"%le\t",nR[i]);
	fprintf(fp,"\t");

	fprintf(fp,"%d\t",iter);
	for(i=0;i<=100;i++)
		fprintf(fp,"%le\t",ia.getY(double(i)/100));
	fprintf(fp,"\t");

	fprintf(fp,"\n");
	fflush(fp);
}


//-------------------------------------------------------------------=72
// End of RRCalibrateProc.cpp
//-------------------------------------------------------------------=72
