//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// rrselect.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Oct/27/1998:	Copied from selectFlat.cpp	
//  Apr/02/1999:	Auto-trimming option was added.	
//
// Bugs:
//
//-------------------------------------------------------------------=72
//	
//	rrselect <infile> <outfile> [<paramfile>]
//	
//	infile:	RRFileListData
//	outfile: RRFileListData
//	paramfile: RRSelectParamData
// 
//	rrselect selects pixels which are reliable data for radiance 
//	recovery. Input images are specified in the <infile>. If the number 
//	of input images is N, one <outfile> and N-1 <outfile>.c.m are 
//	generated. Here, c is number of channel (R, G and B, respectively) 
//	and m is number of image pair (m=0 is the pair of image[0] and 
//	image[1], and so on). <outfile> contains the filenames of 
//	<outfile>.c.m and <outfile>.c.m contain the actual data of selected 
//	pixels. 
//	For information of parameters of this programs, see 
//	RRSelectParamData.h. When <paramfile> is not specified, default
//	values of parameters are used.
//	rrselect returns 0 if it has computed the selected pixels correctly. 
//	Otherwise it returns -1.
//
//-------------------------------------------------------------------=72

#include "RRFiledData.h"
#include "RRFileName.h"
#include "RRNumberedFileName.h"
#include "RRSpatialPixelAverageProc.h"
#include "RRSelectProc.h"
#include "RRSelectAutoTrimProc.h"
#include "RRFileListData.h"
#include "RR2DDoubleArrayData.h"
#include "RRSelectParamData.h"
#include "RRPixelStatisticsData.h"
#include "RRImageFileIO.h"

#include <stdio.h>

const char *tmpTempoFile = "RRSL.tmp.tempo.";
const char *tmpSpatioFile = "RRSL.tmp.spatio.";

RRStatus setFileNames(
			int argc,
			char *argv[],
			RRFileName& infile,
			RRFileName& outfile,
			RRFileName& paramfile)
{
	if(argc==4)
	{
		printf("Infile %s\n",argv[1]);
		infile=RRFileName(argv[1]);
		printf("Outfile %s\n",argv[2]);
		outfile=RRFileName(argv[2]);
		printf("Paramfile %s\n",argv[3]);
		paramfile=RRFileName(argv[3]);
	}
	else if(argc==3)
	{
		printf("Infile %s\n",argv[1]);
		infile=RRFileName(argv[1]);
		printf("Outfile %s\n",argv[2]);
		outfile=RRFileName(argv[2]);
		paramfile=RRFileName("");
	}
	else
	{
		char buf[256];
		printf("open infile\n");
		gets(buf);
		printf("...\n");
		infile=RRFileName(buf);
		printf("open outfile\n");
		gets(buf);
		printf("...\n");
		outfile=RRFileName(buf);
		printf("open paramfile\n");
		gets(buf);
		printf("...\n");
		if(strcmp(buf,"")!=0)
			paramfile=RRFileName(buf);
		else
			paramfile=RRFileName("");
	}

	return RR_SUCCESS;
}

RRStatus loadImagesToPSD(
			const int ch,
			const RRDataTemplate< RRFileListData >& imdata,
			RRNumberedFileName& tempofile)
{
	printf("\tloadImagesToPSD\n");

	const RRFileListData& iad=(const RRFileListData&)(imdata.getSubstance());
	const int numFiles=iad.getNumFiles();

	tempofile.setMaxNumber(numFiles-1);

	for(int i=0;i<numFiles;i++)
	{
		RRIFIODataType channel;
		switch(ch)
		{
		  case 0:
			channel=RRIFIO_DOUBLE_R; // load R channel
			break;
		  case 1:
			channel=RRIFIO_DOUBLE_G; // load G channel
			break;
		  case 2:
			channel=RRIFIO_DOUBLE_B; // load B channel
			break;
		}
		
		RRField< double > mean;
		// Do loading
		if(RRImageFileIO::load(iad.getFileName(i),
		                       RRIFIO_EXT,
							   channel,
							   mean)==RR_ERROR)
			return RR_ERROR;

		RRPixelStatisticsData psd;

		int xs,ys;
		mean.getSize(xs,ys);
		RRField< double > var(xs,ys,0.);
		psd.setAllMeanAndVariance(mean,var);
		psd.setNumSamples(1);

		FILE *fpt=fopen(tempofile.getFileName(i)(0),"wb");
		if(psd.saveBinary(fpt)==RR_ERROR)
			return RR_ERROR;

		fclose(fpt);
	}	

	return RR_SUCCESS;
}

RRStatus spatialAverage(
			const int imgnum,
			RRDataTemplate< RRSelectParamData >& param,
			const RRNumberedFileName& tempofile,
			const RRNumberedFileName& spatiofile)
{
	printf("\tspatialAverage\n");

	FILE *fpt=fopen(&(tempofile.getFileName(imgnum)),"rb");
	RRDataTemplate< RRPixelStatisticsData > tempo(tempofile.getFileName(imgnum)(0));
	RRPixelStatisticsData& psd=(RRPixelStatisticsData&)tempo.getSubstance();
	if(psd.loadBinary(fpt)==RR_ERROR)
		return RR_ERROR;

	FILE *fps=fopen(&(spatiofile.getFileName(imgnum)),"wb");
	RRDataTemplate< RRPixelStatisticsData > spatio(spatiofile.getFileName(imgnum)(0));

	RRArray< RRData* > indata(2);
	indata[0]= &tempo;
	indata[1]= &param;

	RRSpatialPixelAverageProc process(indata,&spatio);
	if(process.doIt()==RR_ERROR)
		return RR_ERROR;

	const RRPixelStatisticsData& fd=(const RRPixelStatisticsData&)spatio.getSubstance();
	if(fd.saveBinary(fps)==RR_ERROR)
		return RR_ERROR;

	fclose(fpt);
	fclose(fps);

	return RR_SUCCESS;
}

bool isRequiredAutoTrim(
			RRDataTemplate< RRSelectParamData >& param)
{
	RRSelectParamData& p=(RRSelectParamData&)param.getSubstance();
	return (p.getTrimRatio()==0)?true:false;
}

RRStatus selectFlat(
			const int ch,
			const RRNumberedFileName& tempofile,
			const RRNumberedFileName& spatiofile,
			RRDataTemplate< RRSelectParamData >& param,
			RRArray< RRDataTemplate< RR2DDoubleArrayData > >& dary)
{
	printf("\tselectFlat\n");

	for(int i=0;i<dary.getSize();i++)
	{
		RRArray< RRDataTemplate< RRPixelStatisticsData > > tempo(2);
		RRArray< RRDataTemplate< RRPixelStatisticsData > > spatio(2);
		for(int n=0;n<2;n++)
		{
			FILE *fpt=fopen(&(tempofile.getFileName(i+n)),"rb");
			RRPixelStatisticsData& psdt=(RRPixelStatisticsData&)tempo[n].getSubstance();
			if(psdt.loadBinary(fpt)==RR_ERROR)
				return RR_ERROR;
			fclose(fpt);

			FILE *fps=fopen(&(spatiofile.getFileName(i+n)),"rb");
			RRPixelStatisticsData& psds=(RRPixelStatisticsData&)spatio[n].getSubstance();
			if(psds.loadBinary(fps)==RR_ERROR)
				return RR_ERROR;
			fclose(fps);
		}

		RRArray< RRData* > indata(5);
		indata[0]= &tempo[0];
		indata[1]= &tempo[1];
		indata[2]= &spatio[0];
		indata[3]= &spatio[1];
		indata[4]= &param;
	
		if(isRequiredAutoTrim(param)==true)
		{
			//printf("AutoTrim is selected.\n");
			RRSelectAutoTrimProc process(indata,&(dary[i]));
			if(process.doIt()==RR_ERROR)
				return RR_ERROR;
		}
		else
		{
			RRSelectProc process(indata,&(dary[i]));
			if(process.doIt()==RR_ERROR)
				return RR_ERROR;
		}
	}

	return RR_SUCCESS;
}

RRStatus saveToOutFiles(
			const int channel,
			const RRArray< RRDataTemplate< RR2DDoubleArrayData > >& dary,
			RRDataTemplate< RRFileListData >& outdata)
{
	int numfiles=dary.getSize();

	RRFileListData& flist=(RRFileListData&)outdata.getSubstance();
	int fnum=flist.getNumFiles();
	flist.setNumFiles(fnum+numfiles);

	char tmp[256];
	strcpy(tmp,outdata.getID().getIndividual());
	RRNumberedFileName prepos(2,strcat(tmp,"."),"."); // channel 0,1,2
	RRNumberedFileName imaryfile(numfiles-1,&(prepos.getFileName(channel)),"");
	
	// copy dary to imary
	for(int i=0;i<numfiles;i++)
	{
		flist.setFileName(fnum+i,&(imaryfile.getFileName(i)));

		RRWOFiledData< RR2DDoubleArrayData > imary(&(imaryfile.getFileName(i)));
		const RR2DDoubleArrayData& da=(const RR2DDoubleArrayData&)dary[i].getSubstance();
		RR2DDoubleArrayData& ia=(RR2DDoubleArrayData&)imary.getSubstance();
		ia=da;
	}

	return RR_SUCCESS;
}

int main(
			int argc,
			char *argv[])
{
	RRFileName infile,outfile,paramfile;

	if(setFileNames(argc,argv,infile,outfile,paramfile)==RR_ERROR)
		return -1;

	RRROFiledData< RRFileListData > imdata(&infile);

	RRDataTemplate< RRSelectParamData > padata(&paramfile); // create default
	if(!(paramfile==RRFileName("")))
		padata=RRROFiledData< RRSelectParamData >(&paramfile);

	RRWOFiledData< RRFileListData > outdata(&outfile);
	RRFileListData& flist=(RRFileListData&)outdata.getSubstance();
	flist.setNumFiles(0);
	
	// process for each channel
	for(int ch=0;ch<3;ch++)
	{
		printf("processing channel %d\n",ch);

		RRNumberedFileName tempofile(0,tmpTempoFile,"");

		if(loadImagesToPSD(ch,imdata,tempofile)==RR_ERROR)
			return -1;

		const int numImg = tempofile.getMaxNumber()+1;
		RRNumberedFileName spatiofile(numImg-1,tmpSpatioFile,"");

		for(int i=0;i<numImg;i++)
			if(spatialAverage(i,padata,tempofile,spatiofile)==RR_ERROR)
				return -1;
	
		RRArray< RRDataTemplate< RR2DDoubleArrayData > > dary(numImg-1);
		if(selectFlat(ch,tempofile,spatiofile,padata,dary)==RR_ERROR)
			return -1;

		if(saveToOutFiles(ch,dary,outdata)==RR_ERROR)
			return -1;

		// clean temporal files.
		for(int n=0;n<numImg;n++)
		{
			if(remove(&(tempofile.getFileName(n)))!=0)
				return -1;
			if(remove(&(spatiofile.getFileName(n)))!=0)
				return -1;
		}
	}	

	printf("Process successfully ended.\n");
	return 0;
}

//-------------------------------------------------------------------=72
// End of rrselect.cpp
//-------------------------------------------------------------------=72
