//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// rrcombine.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Nov/21/1998:	Created	
//
// Bugs:
//
//-------------------------------------------------------------------=72
//	
//	rrselect <infile1> <infile2> <outfile> [<paramfile>]
//	
//	infile1:	RRFileListData -- image files
//	infile2:	RRFileListData -- transfer function files
//	outfile:	RRFileListData -- radiance data files
//	paramfile:	RRCombineParamData
// 
//	rrcombine makes a radiance map image by combining input images.
//	Input images are specified in the <infile1>. rrcombine needs another
//	input <infile2> as the responce function curve data of the imaging
//	system that glabbed the input images. Use rrcalibrate to make 
//	<infile2>. rrcombine outputs one image file named <outfile>.
//	For information of parameters of this program, see 
//	RRCombineParamData.h. When <paramfile> is not specified, default
//	values of parameters are used.
//	rrcombine returns 0 if it has computed a radiance map correctly.
//	Otherwise it returns -1.
//
//-------------------------------------------------------------------=72

#include "RRFiledData.h"
#include "RRFileName.h"
#include "RRNumberedFileName.h"
#include "RRCombineProc.h"
#include "RRFileListData.h"
#include "RRCombineParamData.h"
#include "RRDoubleFieldData.h"
#include "RRImageFileIO.h"

#include <stdio.h>
#include <math.h>

#define RRCM_DEBUG

const char *tmpRadianceFile = "RRCB.tmp.";
const double bestMLuminance = 0.7;

RRStatus setFileNames(
			int argc,
			char *argv[],
			RRFileName& infile1,
			RRFileName& infile2,
			RRFileName& outfile,
			RRFileName& paramfile)
{
	if(argc==5)
	{
		printf("Infile1 %s\n",argv[1]);
		infile1=RRFileName(argv[1]);
		printf("Infile2 %s\n",argv[2]);
		infile2=RRFileName(argv[2]);
		printf("Outfile %s\n",argv[3]);
		outfile=RRFileName(argv[3]);
		printf("Paramfile %s\n",argv[4]);
		paramfile=RRFileName(argv[4]);
	}
	else if(argc==4)
	{
		printf("Infile1 %s\n",argv[1]);
		infile1=RRFileName(argv[1]);
		printf("Infile2 %s\n",argv[2]);
		infile2=RRFileName(argv[2]);
		printf("Outfile %s\n",argv[3]);
		outfile=RRFileName(argv[3]);
		paramfile=RRFileName("");
	}
	else
	{
		char buf[256];
		printf("open infile1\n");
		gets(buf);
		printf("...\n");
		infile1=RRFileName(buf);
		printf("open infile2\n");
		gets(buf);
		printf("...\n");
		infile2=RRFileName(buf);
		printf("open outfile\n");
		gets(buf);
		printf("...\n");
		outfile=RRFileName(buf);
		printf("open paramfile\n");
		gets(buf);
		printf("...\n");
		if(strcmp(buf,"")!=0)
			paramfile=RRFileName(buf);
		else
			paramfile=RRFileName("");
	}

	return RR_SUCCESS;
}

RRStatus loadImagesToField(
			const int ch,
			const RRDataTemplate< RRFileListData >& imgdata,
			RRArray< RRDataTemplate< RRDoubleFieldData > >& img)
{
	printf("\tloadImagesToField\n");

	const RRFileListData& imd = (const RRFileListData&)(imgdata.getSubstance());
	const int numFiles = imd.getNumFiles();

	img.setSize(numFiles);

	for(int i=0;i<numFiles;i++)
	{
		RRIFIODataType channel;
		switch(ch)
		{
		  case 0:
			channel=RRIFIO_DOUBLE_R; // load R channel
			break;
		  case 1:
			channel=RRIFIO_DOUBLE_G; // load G channel
			break;
		  case 2:
			channel=RRIFIO_DOUBLE_B; // load B channel
			break;
		}
		
		RRField< double > fld;
		// Do loading
		if(RRImageFileIO::load(imd.getFileName(i),
		                       RRIFIO_EXT,
							   channel,
							   fld)==RR_ERROR)
			return RR_ERROR;

		RRDoubleFieldData& fdata=(RRDoubleFieldData&)(img[i].getSubstance());
		fdata.setData(fld);
	}	
	return RR_SUCCESS;
}

RRStatus loadCurve(
			const int ch,
			const RRDataTemplate< RRFileListData >& crvdata,
			RRDataTemplate< RRIaCurveData >& crv)
{
	printf("\tloadCurve\n");

	const RRFileListData& crvd = (const RRFileListData&)(crvdata.getSubstance());

	RRROFiledData< RRIaCurveData > fcrv(crvd.getFileName(ch));
	const RRIaCurveData& fc = (const RRIaCurveData&)fcrv.getSubstance();
	RRIaCurveData& cv = (RRIaCurveData&)crv.getSubstance();
	cv=fc;

	return RR_SUCCESS;
}

RRStatus makeMData(
			const RRDataTemplate< RRFileListData >& imgdata,
			RRField< RRPixelD >& mimg)
{
	printf("\tmakeMData\n");

	const RRFileListData& imd = (const RRFileListData&)imgdata.getSubstance();
	const int numFiles = imd.getNumFiles();

	int xs,ys;
	{
		RRField< RRPixelD > img;
		if(RRImageFileIO::load(imd.getFileName(0),
		                       RRIFIO_EXT,
							   RRIFIO_PIXELD,
							   img)==RR_ERROR)
			return RR_ERROR;
		img.getSize(xs,ys);
		mimg=img;
	}

	for(int i=1;i<numFiles;i++)
	{
		RRField< RRPixelD > img;
		if(RRImageFileIO::load(imd.getFileName(i),
		                       RRIFIO_EXT,
							   RRIFIO_PIXELD,
							   img)==RR_ERROR)
			return RR_ERROR;

		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				RRPixelD p=img.getElem(x,y);
				double lump=(p.getR()+p.getG()+p.getB())/3.;
				RRPixelD q=mimg.getElem(x,y);
				double lumq=(q.getR()+q.getG()+q.getB())/3.;
				
				if(p.getR()>0.010 && p.getR()<0.990 &&
				   p.getG()>0.010 && p.getG()<0.990 &&
				   p.getB()>0.010 && p.getB()<0.990 &&
				   fabs(lump-bestMLuminance)<fabs(lumq-bestMLuminance))
					mimg.setElem(x,y,p);
			}
	}

#ifdef RRCM_DEBUG
	RRField< RRPixelD > nmimg(xs,ys,RRPixelD(0,0,0,1));
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD q=mimg.getElem(x,y);
			{
				double mm=(q.getR()+q.getG()+q.getB())/3.;
				if(fabs(q.getR()-mm)/mm>0.02 ||
				   fabs(q.getG()-mm)/mm>0.02 ||
				   fabs(q.getB()-mm)/mm>0.02)
					continue;
			}
			RRPixelD p;
			double r=(2.*q.getR()-q.getG()-q.getB())/3.;
			double g=(-q.getR()+2.*q.getG()-q.getB())/3.;
			double b=(-q.getR()-q.getG()+2.*q.getB())/3.;
			double len=sqrt(r*r+g*g+b*b);
			//p.setR(r/len*0.5+0.5);
			//p.setG(g/len*0.5+0.5);
			//p.setB(b/len*0.5+0.5);
			p.setR(r+0.5);
			p.setG(g+0.5);
			p.setB(b+0.5);
			p.setA(1.);
			nmimg.setElem(x,y,q);
		}
	RRImageFileIO::save(nmimg,RRIFIO_PIXELD,RRIFIO_EXT,"RRCM.debug.mimg.bmp");
#endif // RRCM_DEBUG

	return RR_SUCCESS;
} 

RRStatus makeIData(
			const RRNumberedFileName& radfile,
			RRField< RRPixelD >& iimg)
{
	printf("\tmakeIData\n");

	int xs,ys;
	{
		FILE *fp=fopen(&(radfile.getFileName(0)),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		fld.getSize(xs,ys);
		iimg.setSize(xs,ys);
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setR(v);
				iimg.setElem(x,y,p);
			}
	}

	{
		FILE *fp=fopen(&(radfile.getFileName(1)),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setG(v);
				iimg.setElem(x,y,p);
			}
	}

	{
		FILE *fp=fopen(&(radfile.getFileName(2)),"rb");
		RRDoubleFieldData fdata;
		fdata.loadBinary(fp);
		fclose(fp);

		const RRField< double >& fld=fdata.getData();
		for(int y=0;y<ys;y++)
			for(int x=0;x<xs;x++)
			{
				double v=fld.getElem(x,y);
				RRPixelD p=iimg.getElem(x,y);
				p.setB(v);
				iimg.setElem(x,y,p);
			}
	}

#ifdef RRCM_DEBUG
	RRField< RRPixelD > niimg(xs,ys);
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD q=iimg.getElem(x,y);
			RRPixelD p;
			double r=(2.*q.getR()-q.getG()-q.getB())/3.;
			double g=(-q.getR()+2.*q.getG()-q.getB())/3.;
			double b=(-q.getR()-q.getG()+2.*q.getB())/3.;
			double len=sqrt(r*r+g*g+b*b);
			//p.setR(r/len*0.5+0.5);
			//p.setG(g/len*0.5+0.5);
			//p.setB(b/len*0.5+0.5);
			p.setR(r+0.5);
			p.setG(g+0.5);
			p.setB(b+0.5);
			p.setA(1.);
			niimg.setElem(x,y,q);
		}
	RRImageFileIO::save(niimg,RRIFIO_PIXELD,RRIFIO_EXT,"RRCM.debug.iimg.bmp");
#endif // RRCM_DEBUG

	return RR_SUCCESS;
} 

/* old
RRStatus calcScales(
			const RRField< RRPixelD >& mimg,
			const RRField< RRPixelD >& iimg,
			RRArray< double >& k, //k[3]
			RRField< RRPixelD >& cbal)
{
	printf("\tcalcScales\n");

	const double trim = 0.7;

	int xs,ys;
	iimg.getSize(xs,ys);

	const int left = int(xs-xs*trim)/2; 
	const int bottom = int(ys-ys*trim)/2; 
	const int right = xs-left; 
	const int top = ys-bottom; 

	RRArray< double > iaia(3);
	RRArray< double > iaim(3);
	for(int i=0;i<3;i++)
	{
		iaia[i]=iaim[i]=0.;
	}

	for(int y=bottom;y<top;y++)
		for(int x=left;x<right;x++)
		{
			RRPixelD ia=iimg.getElem(x,y);
			RRPixelD im=mimg.getElem(x,y);

			iaia[0]+=ia.getR()*ia.getR();
			iaia[1]+=ia.getG()*ia.getG();
			iaia[2]+=ia.getB()*ia.getB();
			iaim[0]+=ia.getR()*im.getR();
			iaim[1]+=ia.getG()*im.getG();
			iaim[2]+=ia.getB()*im.getB();
		}

	double kmax=0;
	for(i=0;i<3;i++)
	{
		k[i]=iaim[i]/iaia[i];
		kmax=(k[i]>kmax)?k[i]:kmax;
	}

	for(i=0;i<3;i++)
		k[i]/=kmax;
	printf("k = (%le\t%le\t%le)\n",k[0],k[1],k[2]);

	cbal.setSize(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=iimg.getElem(x,y);
			RRPixelD q(p.getR()*k[0],p.getG()*k[1],p.getB()*k[2],1.);
			cbal.setElem(x,y,q);
		}

#ifdef RRCM_DEBUG
	RRField< RRPixelD > nbimg(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD q=cbal.getElem(x,y);
			RRPixelD p;
			double r=(2.*q.getR()-q.getG()-q.getB())/3.;
			double g=(-q.getR()+2.*q.getG()-q.getB())/3.;
			double b=(-q.getR()-q.getG()+2.*q.getB())/3.;
			double len=sqrt(r*r+g*g+b*b);
			//p.setR(r/len*0.5+0.5);
			//p.setG(g/len*0.5+0.5);
			//p.setB(b/len*0.5+0.5);
			p.setR(r+0.5);
			p.setG(g+0.5);
			p.setB(b+0.5);
			p.setA(1.);
			nbimg.setElem(x,y,p);
		}
	RRImageFileIO::save(nbimg,RRIFIO_PIXELD,RRIFIO_EXT,"RRCM.debug.bimg.bmp");
#endif // RRCM_DEBUG

	return RR_SUCCESS;
}
*/

/**/
RRStatus calcScales(
			const RRField< RRPixelD >& mimg,
			const RRField< RRPixelD >& iimg,
			RRArray< double >& k, //k[3]
			RRField< RRPixelD >& cbal)
{
	const double trim = 0.7;

	int xs,ys;
	iimg.getSize(xs,ys);

	const int left = int(xs-xs*trim)/2; 
	const int bottom = int(ys-ys*trim)/2; 
	const int right = xs-left; 
	const int top = ys-bottom; 

	double AA(0),AB(0),BB(0),CA(0),CB(0);

	for(int y=bottom;y<top;y++)
		for(int x=left;x<right;x++)
		{
			RRPixelD ia=iimg.getElem(x,y);

			//double ialen=sqrt(ia.getR()*ia.getR()+ia.getG()*ia.getG()+ia.getB()*ia.getB());
			//ialen=(ialen==0)?1:ialen;
			//ia.setR(ia.getR()/ialen);
			//ia.setG(ia.getG()/ialen);
			//ia.setB(ia.getB()/ialen);

			RRPixelD im=mimg.getElem(x,y);

			//{
			//	double mm=(im.getR()+im.getG()+im.getB())/3.;
			//	if(fabs(im.getR()-mm)/mm>0.01 ||
			//	   fabs(im.getG()-mm)/mm>0.01 ||
			//	   fabs(im.getB()-mm)/mm>0.01)
			//		continue;
			//}

			//double imlen=sqrt(im.getR()*im.getR()+im.getG()*im.getG()+im.getB()*im.getB());
			//imlen=(imlen==0)?1:imlen;
			//im.setR(im.getR()/imlen);
			//im.setG(im.getG()/imlen);
			//im.setB(im.getB()/imlen);

			double IgMrMb=ia.getG()*(im.getR()-im.getB());
			double IbMgMr=ia.getB()*(im.getG()-im.getR());
			double IrMgMb=ia.getR()*(im.getG()-im.getB());

			AA+=IgMrMb*IgMrMb;
			AB+=IgMrMb*IbMgMr;
			BB+=IbMgMr*IbMgMr;
			CA+=IrMgMb*IgMrMb;
			CB+=IrMgMb*IbMgMr;
		}

	double denom=AA*BB-AB*AB;
	printf("denom = %le\n",denom);
	if(denom==0)
		return RR_ERROR;

	double fracG=BB*CA-AB*CB;
	printf("fracG = %le\n",fracG);
	if(fracG==0)
		return RR_ERROR;
	
	double fracB= -AB*CA+AA*CB;
	printf("fracB = %le\n",fracB);
	if(fracB==0)
		return RR_ERROR;

	k[0]=1.;
	k[1]=fracG/denom;
	k[2]=fracB/denom;

	double kmax=0;
	for(int i=0;i<3;i++)
		kmax=(k[i]>kmax)?k[i]:kmax;

	for(i=0;i<3;i++)
		k[i]/=kmax;
	printf("k = (%le\t%le\t%le)\n",k[0],k[1],k[2]);

	cbal.setSize(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=iimg.getElem(x,y);
			RRPixelD q(p.getR()*k[0],p.getG()*k[1],p.getB()*k[2],1.);
			cbal.setElem(x,y,q);
		}

#ifdef RRCM_DEBUG
	RRField< RRPixelD > nbimg(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD q=cbal.getElem(x,y);
			RRPixelD p;
			double r=(2.*q.getR()-q.getG()-q.getB())/3.;
			double g=(-q.getR()+2.*q.getG()-q.getB())/3.;
			double b=(-q.getR()-q.getG()+2.*q.getB())/3.;
			double len=sqrt(r*r+g*g+b*b);
			//p.setR(r/len*0.5+0.5);
			//p.setG(g/len*0.5+0.5);
			//p.setB(b/len*0.5+0.5);
			p.setR(r+0.5);
			p.setG(g+0.5);
			p.setB(b+0.5);
			p.setA(1.);
			nbimg.setElem(x,y,p);
		}
	RRImageFileIO::save(nbimg,RRIFIO_PIXELD,RRIFIO_EXT,"RRCM.debug.bimg.bmp");
#endif // RRCM_DEBUG

	return RR_SUCCESS;
}
/**/

///* M/|M| = I/|I|
RRStatus calcWScales(
			const RRField< RRPixelD >& mimg,
			const RRField< RRPixelD >& iimg,
			RRArray< double >& k, //k[3]
			RRField< RRPixelD >& cbal)
{
	const double trim = 0.7;

	double IrgMrg(0),IrbMrb(0),IggMrr(0),IggMbb(0),IbbMrr(0),IbbMgg(0),IgbMgb(0);

	int xs,ys;
	iimg.getSize(xs,ys);

	const int left = int(xs-xs*trim)/2; 
	const int bottom = int(ys-ys*trim)/2; 
	const int right = xs-left; 
	const int top = ys-bottom; 

	int cnt=0;
	for(int y=bottom;y<top;y++)
		for(int x=left;x<right;x++)
		{
			RRPixelD ia=iimg.getElem(x,y);
			RRPixelD im=mimg.getElem(x,y);

			{
				double mm=(im.getR()+im.getG()+im.getB())/3.;
				if(fabs(im.getR()-mm)/mm>0.02 ||
				   fabs(im.getG()-mm)/mm>0.02 ||
				   fabs(im.getB()-mm)/mm>0.02)
					continue;
			}

			IrgMrg+=ia.getR()*ia.getG()*im.getR()*im.getG();
			IrbMrb+=ia.getR()*ia.getB()*im.getR()*im.getB();
			IggMrr+=ia.getG()*ia.getG()*im.getR()*im.getR();
			IggMbb+=ia.getG()*ia.getG()*im.getB()*im.getB();
			IbbMrr+=ia.getB()*ia.getB()*im.getR()*im.getR();
			IbbMgg+=ia.getB()*ia.getB()*im.getG()*im.getG();
			IgbMgb+=ia.getG()*ia.getB()*im.getG()*im.getB();
			cnt++;
		}
	printf("cnt = %d\n",cnt);

	double denom=(IggMrr+IggMbb)*(IbbMrr+IbbMgg)-IgbMgb*IgbMgb;
	double fracG=IrgMrg*(IbbMrr+IbbMgg)+IrbMrb*IgbMgb;
	double fracB=IrbMrb*(IggMrr+IggMbb)+IrgMrg*IgbMgb;

	k[0]=1.;
	k[1]=fracG/denom;
	k[2]=fracB/denom;

	double kmax=0;
	for(int i=0;i<3;i++)
		kmax=(k[i]>kmax)?k[i]:kmax;

	for(i=0;i<3;i++)
		k[i]/=kmax;
	printf("k = (%le\t%le\t%le)\n",k[0],k[1],k[2]);

	cbal.setSize(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=iimg.getElem(x,y);
			RRPixelD q(p.getR()*k[0],p.getG()*k[1],p.getB()*k[2],1.);
			cbal.setElem(x,y,q);
		}

#ifdef RRCM_DEBUG
	RRField< RRPixelD > nbimg(xs,ys);
	for(y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD q=cbal.getElem(x,y);
			double len=sqrt(q.getR()*q.getR()+q.getG()*q.getG()+q.getB()*q.getB());
			len=(len==0)?1:len;
			nbimg.setElem(x,y,RRPixelD(q.getR()/len,q.getG()/len,q.getB()/len,1.));
		}
	RRImageFileIO::save(nbimg,RRIFIO_PIXELD,RRIFIO_EXT,"RRCM.debug.bimg.bmp");
#endif // RRCM_DEBUG

	return RR_SUCCESS;
}
//*/

int main(
			int argc,
			char *argv[])
{
	const int numCh = 3;

	RRFileName infile1,infile2,outfile,paramfile;

	if(setFileNames(argc,argv,infile1,infile2,outfile,paramfile)==RR_ERROR)
		return -1;

	RRROFiledData< RRFileListData > imgdata(&infile1);
	RRROFiledData< RRFileListData > crvdata(&infile2);

	RRDataTemplate< RRCombineParamData > padata(&paramfile); // create default
	if(!(paramfile==RRFileName("")))
		padata=RRROFiledData< RRCombineParamData >(&paramfile);

	RRNumberedFileName radfile(numCh-1,tmpRadianceFile,"");

	// process for each channel
	for(int ch=0;ch<numCh;ch++)
	{
		printf("processing channel %d\n",ch);

		RRArray< RRDataTemplate< RRDoubleFieldData > > img;
		if(loadImagesToField(ch,imgdata,img)==RR_ERROR)
			return -1;

		RRDataTemplate< RRIaCurveData > crv;
		if(loadCurve(ch,crvdata,crv)==RR_ERROR)
			return -1;
		
		const int numImg = img.getSize();
		RRArray< RRData* > indata(numImg+2);
		indata[0]= &crv;
		for(int i=0;i<numImg;i++)
			indata[i+1]= &(img[i]);
		indata[numImg+1]= &padata;

		RRDataTemplate< RRDoubleFieldData > raddata;

		RRCombineProc process(indata,&raddata);
		if(process.doIt()==RR_ERROR)
			return -1;

		// temporarily save intermediate data
		FILE *fp=fopen(&(radfile.getFileName(ch)),"wb");
		const RRDoubleFieldData& rfld= (const RRDoubleFieldData&)raddata.getSubstance();
		RRStatus status=rfld.saveBinary(fp);
		fclose(fp);
		if(status==RR_ERROR)
			return -1;
	}	

	// color balancing
	RRField< RRPixelD > cbal;
	{
		RRField< RRPixelD > mimg;
		if(makeMData(imgdata,mimg)==RR_ERROR)
			return -1;

		RRField< RRPixelD > iimg;
		if(makeIData(radfile,iimg)==RR_ERROR)
			return -1;

		RRArray< double > wk(3);
		//RRField< RRPixelD > wbal;
		if(calcWScales(mimg,iimg,wk,cbal)==RR_ERROR)
			return -1;

		//RRArray< double > k(3);
		//if(calcWScales(mimg,wbal,k,cbal)==RR_ERROR)
		//	return -1;
	}

	if(RRImageFileIO::save(cbal,
						   RRIFIO_PIXELD,
						   RRIFIO_EXT,
						   &outfile)==RR_ERROR)
			return -1;

	// clean temporal files.
	for(ch=0;ch<numCh;ch++)
		if(remove(&(radfile.getFileName(ch)))!=0)
			return -1;

	printf("Process successfully ended.\n");
	return 0;
}


//-------------------------------------------------------------------=72
// End of rrcombine.cpp
//-------------------------------------------------------------------=72
