//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// rrcombine.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Nov/21/1998:	Created	
//	Dec/16/1998:	Improved algorithms of combining and color balancing
//
// Bugs:
//
//-------------------------------------------------------------------=72
//	
//	rrselect <infile1> <infile2> <outfile> [<paramfile>]
//	
//	infile1:	RRFileListData -- image files
//	infile2:	RRFileListData -- response curve files
//	outfile:	RRFileListData -- radiance data files
//	paramfile:	RRCombineParamData
// 
//	rrcombine makes a radiance map image by combining input images.
//	Input images are specified in the <infile1>. rrcombine needs another
//	input <infile2> as the responce function curve data of the imaging
//	system that glabbed the input images. Use rrcalibrate to make 
//	<infile2>. rrcombine outputs one image file named <outfile>.
//	For information of parameters of this program, see 
//	RRCombineParamData.h. When <paramfile> is not specified, default
//	values of parameters are used.
//	rrcombine returns 0 if it has computed a radiance map correctly.
//	Otherwise it returns -1.
//
//-------------------------------------------------------------------=72

#include "RRCombineProc.h"
#include "RRColorBalanceProc.h"
#include "RRFileListData.h"
#include "RRCombineParamData.h"
#include "RRFiledData.h"
#include "RRNumberedFileName.h"

#include <stdio.h>
#include <math.h>

//#define RRCM_DEBUG

const char *tmpRadianceFile = "RRCB.tmp.";
const int numCh = 3;

RRStatus setFileNames(
			int argc,
			char *argv[],
			RRFileName& infile1,
			RRFileName& infile2,
			RRFileName& outfile,
			RRFileName& paramfile)
{
	if(argc==5)
	{
		printf("Infile1 %s\n",argv[1]);
		infile1=RRFileName(argv[1]);
		printf("Infile2 %s\n",argv[2]);
		infile2=RRFileName(argv[2]);
		printf("Outfile %s\n",argv[3]);
		outfile=RRFileName(argv[3]);
		printf("Paramfile %s\n",argv[4]);
		paramfile=RRFileName(argv[4]);
	}
	else if(argc==4)
	{
		printf("Infile1 %s\n",argv[1]);
		infile1=RRFileName(argv[1]);
		printf("Infile2 %s\n",argv[2]);
		infile2=RRFileName(argv[2]);
		printf("Outfile %s\n",argv[3]);
		outfile=RRFileName(argv[3]);
		paramfile=RRFileName("");
	}
	else
	{
		char buf[256];
		printf("open infile1\n");
		gets(buf);
		printf("...\n");
		infile1=RRFileName(buf);
		printf("open infile2\n");
		gets(buf);
		printf("...\n");
		infile2=RRFileName(buf);
		printf("open outfile\n");
		gets(buf);
		printf("...\n");
		outfile=RRFileName(buf);
		printf("open paramfile\n");
		gets(buf);
		printf("...\n");
		if(strcmp(buf,"")!=0)
			paramfile=RRFileName(buf);
		else
			paramfile=RRFileName("");
	}

	return RR_SUCCESS;
}

int main(
			int argc,
			char *argv[])
{

try
{
	RRFileName infile1,infile2,outfile,paramfile;

	if(setFileNames(argc,argv,infile1,infile2,outfile,paramfile)==RR_ERROR)
		throw; // return -1;

	// prepare input files
	RRROFiledData< RRFileListData > imgdata(&infile1);
	RRROFiledData< RRFileListData > crvdata(&infile2);

	RRDataTemplate< RRCombineParamData > padata(&paramfile); // create default
	if(!(paramfile==RRFileName("")))
		padata=RRROFiledData< RRCombineParamData >(&paramfile);

	// prepare intermediate files
	RRNumberedFileName radfile(numCh-1,tmpRadianceFile,"");
	RRDataTemplate< RRFileListData > raddata;
	RRFileListData& radlist=(RRFileListData&)raddata.getSubstance();
	radlist.setNumFiles(numCh);
	for(int ch=0;ch<numCh;ch++)
		radlist.setFileName(ch,&(radfile.getFileName(ch)));

	RRArray< RRData* > indata(3);

	// combining process 
	indata[0]= &imgdata;
	indata[1]= &crvdata;
	indata[2]= &padata;
	RRCombineProc process1(indata,&raddata);
	if(process1.doIt()==RR_ERROR)
		throw; // return -1;

	// prepare output file
	RRDataTemplate< RRFileListData > outdata;
	RRFileListData& outlist=(RRFileListData&)outdata.getSubstance();
	outlist.setNumFiles(1);
	outlist.setFileName(0,&outfile);

	// color balancing process
	indata[0]= &imgdata;
	indata[1]= &raddata;
	indata[2]= &padata;
	RRColorBalanceProc process2(indata,&outdata);
	if(process2.doIt()==RR_ERROR)
		throw; // return -1;

	// clean intermediate files.
	for(ch=0;ch<numCh;ch++)
		if(remove(&(radfile.getFileName(ch)))!=0)
			throw; // return -1;
}
catch(...)
{
#ifdef RRCM_DEBUG
	assert(0);
#else
	printf("Process returns error.\n");
	return -1;
#endif // RRCM_DEBUG
}
	printf("Process successfully ended.\n");
	return 0;
}

//-------------------------------------------------------------------=72
// End of rrcombine.cpp
//-------------------------------------------------------------------=72
