//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// rrcalibrate.allch.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Version:              1.0
//
// Modification History:
//  May/03/1999:	Modified from rrcalibrate.cpp 
//
// Bugs:
//  This is a test program for now.
//
// Classes:
//
// Notes:
//	This file defines main() of rrcalibrate.
//
//-------------------------------------------------------------------=72
//	
//	rrcalibrate <infile> <outfile> [<paramfile>]
//	
//	infile:	RRFileListData
//	outfile: RRFileListData
//	paramfile: RRCalibrateParamData
// 
//	rrcalibrate estimates the transfer function of each channel.
//	Imput data, which must be previously generated by rrselect, are 
//	specified in the <infile>. One <outfile> and three <outfile.c> are
//  generated. Here, c is number of channel (R,G and B, respectively).
//	<outfile> contains the filenames of <outfile.c>s and <outfile.c>s 
//	contain the actual data of estimated transfer function and exposure
//  ratios of each channel.	
//	For information of parameters of this programs, see 
//	RRcalibrateParamData.h. When <paramfile> is not specified, default
//	values of parameters are used.
//	rrcalibrate returns 0 if it has computed the transfer function 
//	correctly. Otherwise it returns -1.
//
//-------------------------------------------------------------------=72

#include "RRFiledData.h"
#include "RRFileName.h"
#include "RRNumberedFileName.h"
#include "RRCalibrateProc.h"
#include "RRFileListData.h"
#include "RR2DDoubleArrayData.h"
#include "RRIaCurveData.h"
#include "RRCalibrateParamData.h"

#include <stdio.h>

RRStatus setFileNames(
			int argc,
			char *argv[],
			RRFileName& infile,
			RRFileName& outfile,
			RRFileName& paramfile)
{
	if(argc==4)
	{
		printf("Infile %s\n",argv[1]);
		infile=RRFileName(argv[1]);
		printf("Outfile %s\n",argv[2]);
		outfile=RRFileName(argv[2]);
		printf("Paramfile %s\n",argv[3]);
		paramfile=RRFileName(argv[3]);
	}
	else if(argc==3)
	{
		printf("Infile %s\n",argv[1]);
		infile=RRFileName(argv[1]);
		printf("Outfile %s\n",argv[2]);
		outfile=RRFileName(argv[2]);
		paramfile=RRFileName("");
	}
	else
	{
		char buf[256];
		printf("open infile\n");
		gets(buf);
		printf("...\n");
		infile=RRFileName(buf);
		printf("open outfile\n");
		gets(buf);
		printf("...\n");
		outfile=RRFileName(buf);
		printf("open paramfile\n");
		gets(buf);
		printf("...\n");
		if(strcmp(buf,"")!=0)
			paramfile=RRFileName(buf);
		else
			paramfile=RRFileName("");
	}

	return RR_SUCCESS;
}

RRStatus setDataForChannel(
			const int numChannels,
			RRDataTemplate< RRFileListData >& darydata,
			RRArray< RRDataTemplate< RR2DDoubleArrayData > >&dadata)
{
	printf("\tsetDataForChannel\n");

	const RRFileListData& flist=(const RRFileListData&)darydata.getSubstance();
	const int numDarys = flist.getNumFiles()/numChannels;
	dadata.setSize(numDarys);

	for(int c=0;c<numChannels;c++)
	{
		const int fNum = c*numDarys;

		// copy RR2DDoubleArrayData from file.
		for(int i=0;i<numDarys;i++)
		{
			RRROFiledData< RR2DDoubleArrayData > dafile(flist.getFileName(fNum+i));
			const RR2DDoubleArrayData& fda=(const RR2DDoubleArrayData&)dafile.getSubstance();
			RR2DDoubleArrayData& da=(RR2DDoubleArrayData&)dadata[i].getSubstance();
			
			RRArray< RRArray < double > > datmp=da.getData();
			if(c==0)
				datmp=fda.getData();
			else
			{
				for(int m=0;m<2;m++)
				{
					datmp[m].setSize(datmp[m].getSize()+1);
					datmp[m].paste(datmp[m].getSize()-1,datmp[m].getSize()-1,fda.getData()[m]);
				}
			}
			da.setData(datmp);
		}
	}

	return RR_SUCCESS;
}

RRStatus iaCurveFromN(
			RRArray< RRDataTemplate< RR2DDoubleArrayData > >& dary,
			RRDataTemplate< RRCalibrateParamData >& padata,
			RRDataTemplate< RRIaCurveData >& iadata)
{
	printf("\tiaCurveFromN\n");

	const int numDarys=dary.getSize();
	RRArray< RRData* > indata(numDarys+1);
	for(int i=0;i<numDarys;i++)
		indata[i]= &(dary[i]);
	indata[numDarys]= &padata;

	RRCalibrateProc process(indata,&iadata);
		if(process.doIt()==RR_ERROR)
			return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus saveToOutFiles(
			const int channel,
			RRDataTemplate< RRIaCurveData >& iadata,
			RRDataTemplate< RRFileListData >& outdata)
{
	// set filename to outdata
	RRFileListData& flist=(RRFileListData&)outdata.getSubstance();
	char tmp[256];
	strcpy(tmp,outdata.getID().getIndividual());
	RRNumberedFileName iafile(2,strcat(tmp,"."),""); // channel 0,1,2
	flist.setFileName(channel,&(iafile.getFileName(channel)));
	
	// copy iadata to fiadata
	RRWOFiledData< RRIaCurveData > fiadata(&(iafile.getFileName(channel)));
	const RRIaCurveData& ia=(const RRIaCurveData&)iadata.getSubstance();
	RRIaCurveData& fia=(RRIaCurveData&)fiadata.getSubstance();
	fia=ia;

	return RR_SUCCESS;
}

int main(
			int argc,
			char *argv[])
{
	const numChannels = 3;

	RRFileName infile,outfile,paramfile;

	// obtain input & output filenames
	if(setFileNames(argc,argv,infile,outfile,paramfile)==RR_ERROR)
		return -1;

	// prepare input data
	RRROFiledData< RRFileListData > darydata(&infile);
	const RRFileListData& dflist=(const RRFileListData&)darydata.getSubstance();
	const int numDarys = dflist.getNumFiles()/numChannels;

	// create default parameter
	RRDataTemplate< RRCalibrateParamData > padata(&paramfile);
	RRArray< double > defaultRatio(numDarys);
	for(int i=0;i<numDarys;i++)
		defaultRatio[i]=RRCPD_D_EXPOSURE_RATIO;
	RRCalibrateParamData& param=(RRCalibrateParamData&)padata.getSubstance();
	param.setExposureRatio(defaultRatio);

	// if paramfile is specified, then load param.
	if(!(paramfile==RRFileName("")))  
		padata=RRROFiledData< RRCalibrateParamData >(&paramfile);

	// prepare output data
	RRWOFiledData< RRFileListData > outdata(&outfile);
	RRFileListData& flist=(RRFileListData&)outdata.getSubstance();
	flist.setNumFiles(numChannels);
	
	RRArray< RRDataTemplate< RR2DDoubleArrayData > > dadata(numDarys);
	if(setDataForChannel(numChannels,darydata,dadata)==RR_ERROR)
		return -1;

	RRDataTemplate< RRIaCurveData > iadata;
	if(iaCurveFromN(dadata,padata,iadata)==RR_ERROR)
		return -1;

	for(int ch=0;ch<numChannels;ch++)
		if(saveToOutFiles(ch,iadata,outdata)==RR_ERROR)
			return -1;

	printf("Process successfully ended.\n");
	return 0;
}

//-------------------------------------------------------------------=72
// End of rrcalibrate.cpp
//-------------------------------------------------------------------=72


















