//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRPixelS.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Version:              1.0
//
// Modification History:
//	Nov/22/1998:	Created
//
// Bugs:
//	We assume that ...
//		byte:	unsigned char = 1byte (0 to 255)
//		short:	unsigned short = 2byte (0 to 65,535)
//		double: double = 8byte 
//
//-------------------------------------------------------------------=72

#include "RRPixelS.h"
#include "RRPixelD.h"
#include "RRPixelB.h"

#include <math.h>

const unsigned short RRPS_MAX = 65535;
const unsigned short RRPS_MIN = 0;

unsigned short RRPixelS::convElem(unsigned short e)
{
	return (e>RRPS_MAX)?RRPS_MAX:((e<RRPS_MIN)?RRPS_MIN:e);
}

unsigned short RRPixelS::convElem(unsigned char e)
{
	unsigned short s=(e<<8);
	return (s>RRPS_MAX)?RRPS_MAX:((s<RRPS_MIN)?RRPS_MIN:s);
}

unsigned short RRPixelS::convElem(double e)
{
	e=e*RRPS_MAX+0.5;
	return unsigned short((e>RRPS_MAX)?RRPS_MAX:((e<RRPS_MIN)?RRPS_MIN:e));
}

const unsigned short& RRPixelS::getR()const
{
	return elem[0];
}

void RRPixelS::setR(const unsigned short ir)
{
	elem[0]=convElem(ir);
}

const unsigned short& RRPixelS::getG()const
{
	return elem[1];
}

void RRPixelS::setG(const unsigned short ig)
{
	elem[1]=convElem(ig);
}

const unsigned short& RRPixelS::getB()const
{
	return elem[2];
}

void RRPixelS::setB(const unsigned short ib)
{
	elem[2]=convElem(ib);
}

const unsigned short& RRPixelS::getA()const
{
	return elem[3];
}

void RRPixelS::setA(const unsigned short ia)
{
	elem[3]=convElem(ia);
}

RRPixelS::RRPixelS()
{
	elem[0]=RRPS_MIN;
	elem[1]=RRPS_MIN;
	elem[2]=RRPS_MIN;
	elem[3]=RRPS_MIN;
}

RRPixelS::RRPixelS(const unsigned short *p)
{
	elem[0]=convElem(p[0]);
	elem[1]=convElem(p[1]);
	elem[2]=convElem(p[2]);
	elem[3]=convElem(p[3]);
}

RRPixelS::RRPixelS(
			const unsigned short r,
			const unsigned short g,
			const unsigned short b,
			const unsigned short a)
{
	elem[0]=convElem(r);
	elem[1]=convElem(g);
	elem[2]=convElem(b);
	elem[3]=convElem(a);
}

RRPixelS::RRPixelS(const RRPixelS& v)
{
	elem[0]=v.elem[0];
	elem[1]=v.elem[1];
	elem[2]=v.elem[2];
	elem[3]=v.elem[3];
}

RRPixelS::RRPixelS(const RRPixelB& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelS::RRPixelS(const RRPixelD& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelS::~RRPixelS()
{
	elem[0]='r';
	elem[1]='g';
	elem[2]='b';
	elem[3]='a';
}

int RRPixelS::getByteSize(const int np)
{
	if(np<=0)
		return 0;
	else if(np>=4)
		return sizeof(unsigned short)*4;
	else
		return sizeof(unsigned short)*np;
}

void RRPixelS::getRawBits(const int np,void *bits)const
{
	unsigned short *tmp=(unsigned short *)bits;
	for(int i=0;i<np;i++)
	{
		*tmp=elem[i];
		tmp++;
	}
}

void RRPixelS::setRawBits(const int np,const void *bits)
{
	const unsigned short *tmp=(const unsigned short *)bits;
	for(int i=0;i<np;i++)
	{
		elem[i]= *tmp;
		tmp++;
	}
}

const RRPixelS& RRPixelS::operator=(const RRPixelS& v)
{
	elem[0]=v.elem[0];
	elem[1]=v.elem[1];
	elem[2]=v.elem[2];
	elem[3]=v.elem[3];
	return *this;
}

bool operator==(const RRPixelS& v1,const RRPixelS& v2)
{
	return (v1.elem[0]==v2.elem[0] && 
		    v1.elem[1]==v2.elem[1] && 
			v1.elem[2]==v2.elem[2] && 
			v1.elem[3]==v2.elem[3])?true:false;
}

bool operator!=(const RRPixelS& v1,const RRPixelS& v2)
{
	return (v1.elem[0]!=v2.elem[0] || 
		    v1.elem[1]!=v2.elem[1] || 
			v1.elem[2]!=v2.elem[2] || 
			v1.elem[3]!=v2.elem[3])?true:false;
}

RRPixelS operator+(const RRPixelS& v1,const RRPixelS& v2)
{
	return RRPixelS(RRPixelS::convElem(unsigned short(v1.elem[0]+v2.elem[0])),
					 RRPixelS::convElem(unsigned short(v1.elem[1]+v2.elem[1])),
					 RRPixelS::convElem(unsigned short(v1.elem[2]+v2.elem[2])),
					 RRPixelS::convElem(unsigned short(v1.elem[3]+v2.elem[3])));
}

RRPixelS operator-(const RRPixelS& v1,const RRPixelS& v2)
{
	return RRPixelS(RRPixelS::convElem(unsigned short(v1.elem[0]-v2.elem[0])),
					 RRPixelS::convElem(unsigned short(v1.elem[1]-v2.elem[1])),
					 RRPixelS::convElem(unsigned short(v1.elem[2]-v2.elem[2])),
					 RRPixelS::convElem(unsigned short(v1.elem[3]-v2.elem[3])));
}

RRPixelS operator*(const RRPixelS& v,const double& k)
{
	return RRPixelS(RRPixelS::convElem(v.elem[0]*k),
					 RRPixelS::convElem(v.elem[1]*k),
					 RRPixelS::convElem(v.elem[2]*k),
					 RRPixelS::convElem(v.elem[3]*k));	
}

RRPixelS operator*(const double& k,const RRPixelS& v)
{
	return operator*(v,k);
}

//-------------------------------------------------------------------=72
// End of RRPixelS.cpp
//-------------------------------------------------------------------=72
