//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRPixelD.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Oct/27/1998:	Copied from eldPixelD.cpp
//	Nov/22/1998:	Renewed image and pixel classes
//
// Bugs:
//	We assume that ...
//		byte:	unsigned char = 1byte (0 to 255)
//		short:	unsigned short = 2byte (0 to 65,535)
//		double: double = 8byte 
//
//-------------------------------------------------------------------=72

#include "RRPixelD.h"
#include "RRPixelB.h"
#include "RRPixelS.h"

#include <math.h>

const double RRPD_MAX = 1.;
const double RRPD_MIN = 0.;

double RRPixelD::convElem(double e)
{
	return (e>RRPD_MAX)?RRPD_MAX:((e<RRPD_MIN)?RRPD_MIN:e);
}

double RRPixelD::convElem(unsigned char e)
{
	double ee=double(e)/RRPB_MAX;
	return (ee>RRPD_MAX)?RRPD_MAX:((ee<RRPD_MIN)?RRPD_MIN:ee);
}

double RRPixelD::convElem(unsigned short e)
{
	double ee=double(e)/RRPS_MAX;
	return (ee>RRPD_MAX)?RRPD_MAX:((ee<RRPD_MIN)?RRPD_MIN:ee);
}

const double& RRPixelD::getR()const
{
	return elem[0];
}

void RRPixelD::setR(const double ir)
{
	elem[0]=convElem(ir);
}

const double& RRPixelD::getG()const
{
	return elem[1];
}

void RRPixelD::setG(const double ig)
{
	elem[1]=convElem(ig);
}

const double& RRPixelD::getB()const
{
	return elem[2];
}

void RRPixelD::setB(const double ib)
{
	elem[2]=convElem(ib);
}

const double& RRPixelD::getA()const
{
	return elem[3];
}

void RRPixelD::setA(const double ia)
{
	elem[3]=convElem(ia);
}

RRPixelD::RRPixelD()
{
	elem[0]=RRPD_MIN;
	elem[1]=RRPD_MIN;
	elem[2]=RRPD_MIN;
	elem[3]=RRPD_MIN;
}

RRPixelD::RRPixelD(const double *p)
{
	elem[0]=convElem(p[0]);
	elem[1]=convElem(p[1]);
	elem[2]=convElem(p[2]);
	elem[3]=convElem(p[3]);
}

RRPixelD::RRPixelD(
			const double r,
			const double g,
			const double b,
			const double a)
{
	elem[0]=convElem(r);
	elem[1]=convElem(g);
	elem[2]=convElem(b);
	elem[3]=convElem(a);
}

RRPixelD::RRPixelD(const RRPixelD& v)
{
	elem[0]=v.elem[0];
	elem[1]=v.elem[1];
	elem[2]=v.elem[2];
	elem[3]=v.elem[3];
}

RRPixelD::RRPixelD(const RRPixelB& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelD::RRPixelD(const RRPixelS& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelD::~RRPixelD()
{
	elem[0]=HUGE_VAL;
	elem[1]=HUGE_VAL;
	elem[2]=HUGE_VAL;
	elem[3]=HUGE_VAL;
}

int RRPixelD::getByteSize(const int np)
{
	if(np<=0)
		return 0;
	else if(np>=4)
		return sizeof(double)*4;
	else
		return sizeof(double)*np;
}

void RRPixelD::getRawBits(const int np,void *bits)const
{
	double *tmp=(double*)bits;
	for(int i=0;i<np;i++)
	{
		*tmp=elem[i];
		tmp++;
	}
}

void RRPixelD::setRawBits(const int np,const void *bits)
{
	const double *tmp=(const double*)bits;
	for(int i=0;i<np;i++)
	{
		elem[i]= *tmp;
		tmp++;
	}
}

const RRPixelD& RRPixelD::operator=(const RRPixelD& v)
{
	elem[0]=v.elem[0];	
	elem[1]=v.elem[1];	
	elem[2]=v.elem[2];	
	elem[3]=v.elem[3];	
	return *this;
}

bool operator==(const RRPixelD& v1,const RRPixelD& v2)
{
	return (v1.elem[0]==v2.elem[0] && 
		    v1.elem[1]==v2.elem[1] && 
			v1.elem[2]==v2.elem[2] && 
			v1.elem[3]==v2.elem[3])?true:false;
}

bool operator!=(const RRPixelD& v1,const RRPixelD& v2)
{
	return (v1.elem[0]!=v2.elem[0] || 
		    v1.elem[1]!=v2.elem[1] || 
			v1.elem[2]!=v2.elem[2] || 
			v1.elem[3]!=v2.elem[3])?true:false;
}

RRPixelD operator+(const RRPixelD& v1,const RRPixelD& v2)
{
	return RRPixelD(RRPixelD::convElem(v1.elem[0]+v2.elem[0]),
					 RRPixelD::convElem(v1.elem[1]+v2.elem[1]),
					 RRPixelD::convElem(v1.elem[2]+v2.elem[2]),
					 RRPixelD::convElem(v1.elem[3]+v2.elem[3]));
}

RRPixelD operator-(const RRPixelD& v1,const RRPixelD& v2)
{
	return RRPixelD(RRPixelD::convElem(v1.elem[0]-v2.elem[0]),
					 RRPixelD::convElem(v1.elem[1]-v2.elem[1]),
					 RRPixelD::convElem(v1.elem[2]-v2.elem[2]),
					 RRPixelD::convElem(v1.elem[3]-v2.elem[3]));
}

RRPixelD operator*(const RRPixelD& v,const double& k)
{
	return RRPixelD(RRPixelD::convElem(v.elem[0]*k),
					 RRPixelD::convElem(v.elem[1]*k),
					 RRPixelD::convElem(v.elem[2]*k),
					 RRPixelD::convElem(v.elem[3]*k));	
}

RRPixelD operator*(const double& k,const RRPixelD& v)
{
	return operator*(v,k);
}

//-------------------------------------------------------------------=72
// End of RRPixelD.C
//-------------------------------------------------------------------=72
