//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRPixelB.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Version:              1.1
//
// Modification History:
//  Oct/27/1998:	Copied from eldPixelB.cpp
//	Nov/22/1998:	Renewed image and pixel classes
//
// Bugs:
//	We assume that ...
//		byte:	unsigned char = 1byte (0 to 255)
//		short:	unsigned short = 2byte (0 to 65,535)
//		double: double = 8byte 
//
//-------------------------------------------------------------------=72

#include "RRPixelB.h"
#include "RRPixelD.h"
#include "RRPixelS.h"

#include <math.h>

const unsigned char RRPB_MAX = 255;
const unsigned char RRPB_MIN = 0;

unsigned char RRPixelB::convElem(unsigned char e)
{
	return (e>RRPB_MAX)?RRPB_MAX:((e<RRPB_MIN)?RRPB_MIN:e);
}

unsigned char RRPixelB::convElem(unsigned short e)
{
	e=((e&0xFF00)>>8);
	return unsigned char((e>RRPB_MAX)?RRPB_MAX:((e<RRPB_MIN)?RRPB_MIN:e));
}

unsigned char RRPixelB::convElem(double e)
{
	e=e*RRPB_MAX+0.5;
	return unsigned char((e>RRPB_MAX)?RRPB_MAX:((e<RRPB_MIN)?RRPB_MIN:e));
}

const unsigned char& RRPixelB::getR()const
{
	return elem[0];
}

void RRPixelB::setR(const unsigned char ir)
{
	elem[0]=convElem(ir);
}

const unsigned char& RRPixelB::getG()const
{
	return elem[1];
}

void RRPixelB::setG(const unsigned char ig)
{
	elem[1]=convElem(ig);
}

const unsigned char& RRPixelB::getB()const
{
	return elem[2];
}

void RRPixelB::setB(const unsigned char ib)
{
	elem[2]=convElem(ib);
}

const unsigned char& RRPixelB::getA()const
{
	return elem[3];
}

void RRPixelB::setA(const unsigned char ia)
{
	elem[3]=convElem(ia);
}

RRPixelB::RRPixelB()
{
	elem[0]=RRPB_MIN;
	elem[1]=RRPB_MIN;
	elem[2]=RRPB_MIN;
	elem[3]=RRPB_MIN;
}

RRPixelB::RRPixelB(const int p) // assuming int = 4byte
{
	elem[0]=convElem(unsigned char((p>>24)|0xFF));
	elem[1]=convElem(unsigned char((p>>16)|0xFF));
	elem[2]=convElem(unsigned char((p>>8)|0xFF));
	elem[3]=convElem(unsigned char((p)|0xFF));
}

RRPixelB::RRPixelB(const unsigned char *p)
{
	elem[0]=convElem(p[0]);
	elem[1]=convElem(p[1]);
	elem[2]=convElem(p[2]);
	elem[3]=convElem(p[3]);
}

RRPixelB::RRPixelB(
			const unsigned char r,
			const unsigned char g,
			const unsigned char b,
			const unsigned char a)
{
	elem[0]=convElem(r);
	elem[1]=convElem(g);
	elem[2]=convElem(b);
	elem[3]=convElem(a);
}

RRPixelB::RRPixelB(const RRPixelB& v)
{
	elem[0]=v.elem[0];
	elem[1]=v.elem[1];
	elem[2]=v.elem[2];
	elem[3]=v.elem[3];
}

RRPixelB::RRPixelB(const RRPixelS& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelB::RRPixelB(const RRPixelD& v)
{
	elem[0]=convElem(v.getR());
	elem[1]=convElem(v.getG());
	elem[2]=convElem(v.getB());
	elem[3]=convElem(v.getA());
}

RRPixelB::~RRPixelB()
{
	elem[0]='r';
	elem[1]='g';
	elem[2]='b';
	elem[3]='a';
}

int RRPixelB::getByteSize(const int np)
{
	if(np<=0)
		return 0;
	else if(np>=4)
		return sizeof(unsigned char)*4;
	else
		return sizeof(unsigned char)*np;
}

void RRPixelB::getRawBits(const int np,void *bits)const
{
	unsigned char *tmp=(unsigned char *)bits;
	for(int i=0;i<np;i++)
	{
		*tmp=elem[i];
		tmp++;
	}
}

void RRPixelB::setRawBits(const int np,const void *bits)
{
	const unsigned char *tmp=(const unsigned char *)bits;
	for(int i=0;i<np;i++)
	{
		elem[i]= *tmp;
		tmp++;
	}
}

const RRPixelB& RRPixelB::operator=(const RRPixelB& v)
{
	elem[0]=v.elem[0];
	elem[1]=v.elem[1];
	elem[2]=v.elem[2];
	elem[3]=v.elem[3];
	return *this;
}

bool operator==(const RRPixelB& v1,const RRPixelB& v2)
{
	return (v1.elem[0]==v2.elem[0] && 
		    v1.elem[1]==v2.elem[1] && 
			v1.elem[2]==v2.elem[2] && 
			v1.elem[3]==v2.elem[3])?true:false;
}

bool operator!=(const RRPixelB& v1,const RRPixelB& v2)
{
	return (v1.elem[0]!=v2.elem[0] || 
		    v1.elem[1]!=v2.elem[1] || 
			v1.elem[2]!=v2.elem[2] || 
			v1.elem[3]!=v2.elem[3])?true:false;
}

RRPixelB operator+(const RRPixelB& v1,const RRPixelB& v2)
{
	return RRPixelB(RRPixelB::convElem(unsigned char(v1.elem[0]+v2.elem[0])),
					 RRPixelB::convElem(unsigned char(v1.elem[1]+v2.elem[1])),
					 RRPixelB::convElem(unsigned char(v1.elem[2]+v2.elem[2])),
					 RRPixelB::convElem(unsigned char(v1.elem[3]+v2.elem[3])));
}

RRPixelB operator-(const RRPixelB& v1,const RRPixelB& v2)
{
	return RRPixelB(RRPixelB::convElem(unsigned char(v1.elem[0]-v2.elem[0])),
					 RRPixelB::convElem(unsigned char(v1.elem[1]-v2.elem[1])),
					 RRPixelB::convElem(unsigned char(v1.elem[2]-v2.elem[2])),
					 RRPixelB::convElem(unsigned char(v1.elem[3]-v2.elem[3])));
}

RRPixelB operator*(const RRPixelB& v,const double& k)
{
	return RRPixelB(RRPixelB::convElem(v.elem[0]*k),
					 RRPixelB::convElem(v.elem[1]*k),
					 RRPixelB::convElem(v.elem[2]*k),
					 RRPixelB::convElem(v.elem[3]*k));	
}

RRPixelB operator*(const double& k,const RRPixelB& v)
{
	return operator*(v,k);
}

//-------------------------------------------------------------------=72
// End of RRPixelB.cpp
//-------------------------------------------------------------------=72
