//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRImageFileIO.h
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Date:                 Nov/22/1998
// Version:              1.4
//
// Modification History:
//  Oct/27/1998:	Created	
//	Nov/22/1998:	Renewed image and pixel classes
//	Dec/05/1998:	Implemented saving funcions
//  Jan/23/1999:	Bug fix of mfConvertToField()
//  Feb/27/1999:	Bug fix of mfConvertToField(),mfConvertFromField()
//
// Bugs:
//	We assume that ...
//		byte:	unsigned char = 1byte (0 to 255)
//		short:	unsigned short = 2byte (0 to 65,535)
//		double: double = 8byte 
//
//-------------------------------------------------------------------=72
//
//	RRImageFileIO class has utility functions for I/O to/from a few kind
//	of image file formats.
//	For now RRImageFileIO supports bmp and ppm and ppm16 format.
//	ppm16 format is similar to ppm format but has 16bit grayscale (See
//	RRPPM16Image.h).    
// 
//	ex.
//	When you want to load G channel data of a bmp file with double
//	precision, use
//
//	RRField< double > loadToHere; // 2-D array of double.
//	RRImageFileIO::load("filename",RRIFIO_BMP,RRIFIO_DOUBLE_G,loadToHere);
//
//	When you want to save RRField< RRPixelB > saveFromHere data to a ppm 
//	file, use
//
//	RRImageFileIO::save(SaveFromHere,RRIFIO_PIXELB,RRIFIO_PPM,"filename");
//
//-------------------------------------------------------------------=72

#ifndef _RR_IMAGE_FILE_IO_H
#define _RR_IMAGE_FILE_IO_H

#include "RRDef.h"
#include "RRFileName.h"
#include "RRImage.h"
#include "RRPixelD.h"
#include "RRPixelB.h"
#include "RRPixelS.h"

#include <string.h>
#include <stdio.h>

typedef enum
{
	RRIFIO_BMP, // file type is bmp.
	RRIFIO_PPM, // file type is ppm.
	RRIFIO_PPM16, // file type is ppm16.
	RRIFIO_EXT  // file type is same to the file extention.
} RRIFIOFileType;

typedef enum
{
	RRIFIO_PIXELD,		// pixel data is loaded/saved as RRField< RRPixelD >.
	RRIFIO_PIXELB,		// pixel data is loaded/saved as RRField< RRPixelB >.  
	RRIFIO_PIXELS,		// pixel data is loaded/saved as RRField< RRPixelS >.  
	RRIFIO_DOUBLE_R,	// R data is loaded/saved as RRField< double >.
	RRIFIO_DOUBLE_G,	// G data is loaded/saved as RRField< double >.	
	RRIFIO_DOUBLE_B,	// B data is loaded/saved as RRField< double >.
	RRIFIO_DOUBLE_A,	// A data is loaded/saved as RRField< double >.
	RRIFIO_DOUBLE_L,	// Luminance data is loaded/saved as RRField< double >.
	RRIFIO_BYTE_R,		// R data is loaded/saved as RRField< unsigned char >.
	RRIFIO_BYTE_G,		// G data is loaded/saved as RRField< unsigned char >.
	RRIFIO_BYTE_B,		// B data is loaded/saved as RRField< unsigned char >.
	RRIFIO_BYTE_A,		// A data is loaded/saved as RRField< unsigned char >.
	RRIFIO_BYTE_L,		// Luminance data is loaded/saved as RRField< unsigned char >.
	RRIFIO_SHORT_R,		// R data is loaded/saved as RRField< unsigned short >.
	RRIFIO_SHORT_G,		// G data is loaded/saved as RRField< unsigned short >.
	RRIFIO_SHORT_B,		// B data is loaded/saved as RRField< unsigned short >.
	RRIFIO_SHORT_A,		// A data is loaded/saved as RRField< unsigned short >.
	RRIFIO_SHORT_L		// Luminance data is loaded/saved as RRField< unsigned short >.
} RRIFIODataType;

class RRImageFileIO
{
  private:

	static RRStatus mfLoadBMPToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld);
	static RRStatus mfLoadPPMToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld);
	static RRStatus mfLoadPPM16ToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld);

	static RRStatus mfSaveBMPFromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file);
	static RRStatus mfSavePPMFromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file);
	static RRStatus mfSavePPM16FromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file);

	static RRStatus mfLoadToField(
			const RRFileName& file,
			const RRIFIOFileType ftype,
			RRField< RRPixelD >& fld);

	static RRStatus mfSaveFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIOFileType ftype,
			const RRFileName& file);

	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& data);
	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelB >& data);
	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelS >& data);
	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< double >& data);
	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< unsigned char >& data);
	static RRStatus mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< unsigned short >& data);

	static RRStatus mfConvertToField(
			const RRField< RRPixelD >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);
	static RRStatus mfConvertToField(
			const RRField< RRPixelB >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);
	static RRStatus mfConvertToField(
			const RRField< RRPixelS >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);
	static RRStatus mfConvertToField(
			const RRField< double >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);
	static RRStatus mfConvertToField(
			const RRField< unsigned char >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);
	static RRStatus mfConvertToField(
			const RRField< unsigned short >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld);

  public:

	RRImageFileIO(){}
	~RRImageFileIO(){}

	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelB >& data);
	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& data);
	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelS >& data);
	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< double >& data);
	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< unsigned char >& data);
	static RRStatus load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< unsigned short >& data);

	static RRStatus save(
			const RRField< RRPixelB >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
	static RRStatus save(
			const RRField< RRPixelD >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
	static RRStatus save(
			const RRField< RRPixelS >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
	static RRStatus save(
			const RRField< double >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
	static RRStatus save(
			const RRField< unsigned char >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
	static RRStatus save(
			const RRField< unsigned short >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile);
};

#endif // _RR_IMAGE_FILE_IO_H

//-------------------------------------------------------------------=72
// End of RRImageFileIO.h
//-------------------------------------------------------------------=72
