//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRImageFileIO.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
// Date:                 Nov/22/1998
// Version:              1.4
//
// Modification History:
//  Oct/27/1998:	Created	
//	Nov/22/1998:	Renewed image and pixel classes
//	Dec/05/1998:	Implemented saving funcions
//  Jan/23/1999:	Bug fix of mfConvertToField()
//  Feb/27/1999:	Bug fix of mfConvertToField(),mfConvertFromField()
//
// Bugs:
//	We assume that ...
//		byte:	unsigned char = 1byte (0 to 255)
//		short:	unsigned short = 2byte (0 to 65,535)
//		double: double = 8byte 
//
// Classes:
//
// Notes:
//
//-------------------------------------------------------------------=72

#include "RRImageFileIO.h"

#ifdef SUPPORT_BMP
# include "RRBMPImage.h"
#endif // SUPPORT_BMP

#ifdef SUPPORT_PPM
# include "RRPPMImage.h"
#endif // SUPPORT_PPM

#ifdef SUPPORT_PPM16
# include "RRPPM16Image.h"
#endif // SUPPORT_PPM16

RRStatus RRImageFileIO::mfLoadBMPToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld)
{
#ifdef SUPPORT_BMP

	RRROBMPImage img(&file);

	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,RRPixelD(img.getElem(x,y)));
	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_BMP
}

RRStatus RRImageFileIO::mfLoadPPMToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld)
{
#ifdef SUPPORT_PPM

	RRROPPMImage img(&file);
	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,RRPixelD(img.getElem(x,y)));
	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_PPM
}

RRStatus RRImageFileIO::mfLoadPPM16ToField(
			const RRFileName& file,
			RRField< RRPixelD >& fld)
{
#ifdef SUPPORT_PPM16

	RRROPPM16Image img(&file);
	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,RRPixelD(img.getElem(x,y)));
		
	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_PPM16
}

RRStatus RRImageFileIO::mfSaveBMPFromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file)
{
#ifdef SUPPORT_BMP

	RRWOBMPImage img(&file);

	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setElem(x,y,RRPixelB(fld.getElem(x,y)));

	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_BMP
}

RRStatus RRImageFileIO::mfSavePPMFromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file)
{
#ifdef SUPPORT_PPM

	RRWOPPMImage img(&file);

	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setElem(x,y,RRPixelB(fld.getElem(x,y)));
	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_PPM
}

RRStatus RRImageFileIO::mfSavePPM16FromField(
			const RRField< RRPixelD >& fld,
			const RRFileName& file)
{
#ifdef SUPPORT_PPM16

	RRWOPPM16Image img(&file);

	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setElem(x,y,RRPixelS(fld.getElem(x,y)));
	return RR_SUCCESS;	

#else
	
	return RR_ERROR;

#endif // SUPPORT_PPM16
}

RRStatus RRImageFileIO::mfLoadToField(
			const RRFileName& file,
			const RRIFIOFileType ftype,
			RRField< RRPixelD >& fld)
{
	switch(ftype)
	{
	  case RRIFIO_BMP:
		if(mfLoadBMPToField(file,fld)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_PPM:
		if(mfLoadPPMToField(file,fld)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_PPM16:
		if(mfLoadPPM16ToField(file,fld)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_EXT:
		if(strcmp(file.getExtension(),".bmp")==0)
		{
			if(mfLoadBMPToField(file,fld)==RR_ERROR)
				return RR_ERROR;
		}
		else if(strcmp(file.getExtension(),".ppm")==0)
		{
			if(mfLoadPPMToField(file,fld)==RR_ERROR)
				return RR_ERROR;
		}
		else if(strcmp(file.getExtension(),".ppm16")==0)
		{
			if(mfLoadPPM16ToField(file,fld)==RR_ERROR)
				return RR_ERROR;
		}
		else
			return RR_ERROR;
		break;
	}
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfSaveFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIOFileType ftype,
			const RRFileName& file)
{
	switch(ftype)
	{
	  case RRIFIO_BMP:
		if(mfSaveBMPFromField(fld,file)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_PPM:
		if(mfSavePPMFromField(fld,file)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_PPM16:
		if(mfSavePPM16FromField(fld,file)==RR_ERROR)
			return RR_ERROR;
		break;	
	  case RRIFIO_EXT:
		if(strcmp(file.getExtension(),".bmp")==0)
		{
			if(mfSaveBMPFromField(fld,file)==RR_ERROR)
				return RR_ERROR;
		}
		else if(strcmp(file.getExtension(),".ppm")==0)
		{
			if(mfSavePPMFromField(fld,file)==RR_ERROR)
				return RR_ERROR;
		}
		else if(strcmp(file.getExtension(),".ppm16")==0)
		{
			if(mfSavePPM16FromField(fld,file)==RR_ERROR)
				return RR_ERROR;
		}
		else
			return RR_ERROR;
		break;
	}
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& data)
{
	if(dtype!=RRIFIO_PIXELD)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			data.setElem(x,y,p);
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelB >& data)
{
	if(dtype!=RRIFIO_PIXELB)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			data.setElem(x,y,RRPixelB(p));
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< RRPixelS >& data)
{
	if(dtype!=RRIFIO_PIXELS)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			data.setElem(x,y,RRPixelS(p));
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< double >& data)
{
	if(dtype!=RRIFIO_DOUBLE_R &&
	   dtype!=RRIFIO_DOUBLE_G &&
	   dtype!=RRIFIO_DOUBLE_B &&
	   dtype!=RRIFIO_DOUBLE_A &&
	   dtype!=RRIFIO_DOUBLE_L)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);

			switch(dtype)
			{
			  case RRIFIO_DOUBLE_R:
				data.setElem(x,y,p.getR());
				break;
			  case RRIFIO_DOUBLE_G:
				data.setElem(x,y,p.getG());
				break;
			  case RRIFIO_DOUBLE_B:
				data.setElem(x,y,p.getB());
				break;
			  case RRIFIO_DOUBLE_A:
				data.setElem(x,y,p.getA());
				break;
			  case RRIFIO_DOUBLE_L:
				data.setElem(x,y,(p.getR()+p.getG()+p.getB())/3.);
				break;
			}
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< unsigned char >& data)
{
	if(dtype!=RRIFIO_BYTE_R &&
	   dtype!=RRIFIO_BYTE_G &&
	   dtype!=RRIFIO_BYTE_B &&
	   dtype!=RRIFIO_BYTE_A &&
	   dtype!=RRIFIO_BYTE_L)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelB p(fld.getElem(x,y));

			switch(dtype)
			{
			  case RRIFIO_BYTE_R:
				data.setElem(x,y,p.getR());
				break;
			  case RRIFIO_BYTE_G:
				data.setElem(x,y,p.getG());
				break;
			  case RRIFIO_BYTE_B:
				data.setElem(x,y,p.getB());
				break;
			  case RRIFIO_BYTE_A:
				data.setElem(x,y,p.getA());
				break;
			  case RRIFIO_BYTE_L:
				data.setElem(x,y,(p.getR()+p.getG()+p.getB())/3);
				break;
			}
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertFromField(
			const RRField< RRPixelD >& fld,
			const RRIFIODataType dtype,
			RRField< unsigned short >& data)
{
	if(dtype!=RRIFIO_SHORT_R &&
	   dtype!=RRIFIO_SHORT_G &&
	   dtype!=RRIFIO_SHORT_B &&
	   dtype!=RRIFIO_SHORT_A &&
	   dtype!=RRIFIO_SHORT_L)
		return RR_ERROR;

	int xs,ys;
	fld.getSize(xs,ys);
	data.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelS p(fld.getElem(x,y));

			switch(dtype)
			{
			  case RRIFIO_SHORT_R:
				data.setElem(x,y,p.getR());
				break;
			  case RRIFIO_SHORT_G:
				data.setElem(x,y,p.getG());
				break;
			  case RRIFIO_SHORT_B:
				data.setElem(x,y,p.getB());
				break;
			  case RRIFIO_SHORT_A:
				data.setElem(x,y,p.getA());
				break;
			  case RRIFIO_SHORT_L:
				data.setElem(x,y,(p.getR()+p.getG()+p.getB())/3);
				break;
			}
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelB >& data)
{
	if(dtype!=RRIFIO_PIXELB)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& data)
{
	if(dtype!=RRIFIO_PIXELD)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< RRPixelS >& data)
{
	if(dtype!=RRIFIO_PIXELS)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< double >& data)
{
	if(dtype!=RRIFIO_DOUBLE_R &&
	   dtype!=RRIFIO_DOUBLE_G &&
	   dtype!=RRIFIO_DOUBLE_B &&
	   dtype!=RRIFIO_DOUBLE_A &&
	   dtype!=RRIFIO_DOUBLE_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< unsigned char >& data)
{
	if(dtype!=RRIFIO_BYTE_R &&
	   dtype!=RRIFIO_BYTE_G &&
	   dtype!=RRIFIO_BYTE_B &&
	   dtype!=RRIFIO_BYTE_A &&
	   dtype!=RRIFIO_BYTE_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::load(
			const char *imagefile,
			const RRIFIOFileType ftype,
			const RRIFIODataType dtype,
			RRField< unsigned short >& data)
{
	if(dtype!=RRIFIO_SHORT_R &&
	   dtype!=RRIFIO_SHORT_G &&
	   dtype!=RRIFIO_SHORT_B &&
	   dtype!=RRIFIO_SHORT_A &&
	   dtype!=RRIFIO_SHORT_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// data is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;
	  
	if(mfConvertFromField(fld,dtype,data)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< RRPixelD >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld)
{
	if(dtype!=RRIFIO_PIXELD)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=data.getElem(x,y);
			fld.setElem(x,y,p);
		}
	
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< RRPixelB>& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD>& fld)
{
	if(dtype!=RRIFIO_PIXELB)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p(data.getElem(x,y));
			fld.setElem(x,y,p);
		}

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< RRPixelS >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld)
{
	if(dtype!=RRIFIO_PIXELS)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p(data.getElem(x,y));
			fld.setElem(x,y,p);
		}

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< double >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld)
{
	if(dtype!=RRIFIO_DOUBLE_R &&
	   dtype!=RRIFIO_DOUBLE_G &&
	   dtype!=RRIFIO_DOUBLE_B &&
	   dtype!=RRIFIO_DOUBLE_A &&
	   dtype!=RRIFIO_DOUBLE_L)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	int fxs,fys;
	fld.getSize(fxs,fys);
	if(xs!=fxs || ys!=fys)
		fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			double v=data.getElem(x,y);

			switch(dtype)
			{
			  case RRIFIO_DOUBLE_R:
				p.setR(v);
				break;
			  case RRIFIO_DOUBLE_G:
				p.setG(v);
				break;
			  case RRIFIO_DOUBLE_B:
				p.setB(v);
				break;
			  case RRIFIO_DOUBLE_A:
				p.setA(v);
				break;
			  case RRIFIO_DOUBLE_L:
				p.setR(v);
				p.setG(v);
				p.setB(v);
				break;
			}

			fld.setElem(x,y,p);
		}

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< unsigned char >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld)
{
	if(dtype!=RRIFIO_BYTE_R &&
	   dtype!=RRIFIO_BYTE_G &&
	   dtype!=RRIFIO_BYTE_B &&
	   dtype!=RRIFIO_BYTE_A &&
	   dtype!=RRIFIO_BYTE_L)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	int fxs,fys;
	fld.getSize(fxs,fys);
	if(xs!=fxs || ys!=fys)
		fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			unsigned char v=data.getElem(x,y);

			switch(dtype)
			{
			  case RRIFIO_BYTE_R:
				p.setR(double(v)/RRPB_MAX);
				break;
			  case RRIFIO_BYTE_G:
				p.setG(double(v)/RRPB_MAX);
				break;
			  case RRIFIO_BYTE_B:
				p.setB(double(v)/RRPB_MAX);
				break;
			  case RRIFIO_BYTE_A:
				p.setA(double(v)/RRPB_MAX);
				break;
			  case RRIFIO_BYTE_L:
				p.setR(double(v)/RRPB_MAX);
				p.setG(double(v)/RRPB_MAX);
				p.setB(double(v)/RRPB_MAX);
				break;
			}

			fld.setElem(x,y,p);
		}

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::mfConvertToField(
			const RRField< unsigned short >& data,
			const RRIFIODataType dtype,
			RRField< RRPixelD >& fld)
{
	if(dtype!=RRIFIO_SHORT_R &&
	   dtype!=RRIFIO_SHORT_G &&
	   dtype!=RRIFIO_SHORT_B &&
	   dtype!=RRIFIO_SHORT_A &&
	   dtype!=RRIFIO_SHORT_L)
		return RR_ERROR;

	int xs,ys;
	data.getSize(xs,ys);
	int fxs,fys;
	fld.getSize(fxs,fys);
	if(xs!=fxs || ys!=fys)
		fld.setSize(xs,ys);
	
	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			unsigned short v=data.getElem(x,y);

			switch(dtype)
			{
			  case RRIFIO_SHORT_R:
				p.setR(double(v)/RRPS_MAX);
				break;
			  case RRIFIO_SHORT_G:
				p.setG(double(v)/RRPS_MAX);
				break;
			  case RRIFIO_SHORT_B:
				p.setB(double(v)/RRPS_MAX);
				break;
			  case RRIFIO_SHORT_A:
				p.setA(double(v)/RRPS_MAX);
				break;
			  case RRIFIO_SHORT_L:
				p.setR(double(v)/RRPS_MAX);
				p.setG(double(v)/RRPS_MAX);
				p.setB(double(v)/RRPS_MAX);
				break;
			}

			fld.setElem(x,y,p);
		}

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< RRPixelB >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_PIXELB)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;
	  
	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< RRPixelD >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_PIXELD)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< RRPixelS >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_PIXELS)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< double >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_DOUBLE_R &&
	   dtype!=RRIFIO_DOUBLE_G &&
	   dtype!=RRIFIO_DOUBLE_B &&
	   dtype!=RRIFIO_DOUBLE_A &&
	   dtype!=RRIFIO_DOUBLE_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< unsigned char >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_BYTE_R &&
	   dtype!=RRIFIO_BYTE_G &&
	   dtype!=RRIFIO_BYTE_B &&
	   dtype!=RRIFIO_BYTE_A &&
	   dtype!=RRIFIO_BYTE_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRStatus RRImageFileIO::save(
			const RRField< unsigned short >& data,
			const RRIFIODataType dtype,
			const RRIFIOFileType ftype,
			const char *imagefile)
{
	if(dtype!=RRIFIO_SHORT_R &&
	   dtype!=RRIFIO_SHORT_G &&
	   dtype!=RRIFIO_SHORT_B &&
	   dtype!=RRIFIO_SHORT_A &&
	   dtype!=RRIFIO_SHORT_L)
		return RR_ERROR;

	RRField< RRPixelD > fld;
	RRFileName file(imagefile);

	// image is once loaded as RRField< RRPixelD >
	if(mfLoadToField(file,ftype,fld)==RR_ERROR)
		return RR_ERROR;

	// data is once converted to RRField< RRPixelD >
	if(mfConvertToField(data,dtype,fld)==RR_ERROR)
		return RR_ERROR;

	if(mfSaveFromField(fld,ftype,file)==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of RRImageFileIO.cpp
//-------------------------------------------------------------------=72
