//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRBMPImage.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.1
//
// Modification History:
//  Oct/27/1998:	Copied from eldBMPImage.cpp
//	Nov/22/1998:	Renewed image and pixel classes
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "RRBMPImage.h"

#include <stdio.h>

unsigned char *RRBMPImage::loadDIBitmap(const char	*filename, 
                                         BITMAPINFO	**info)
{
	FILE			*fp;	
	unsigned char	*bits;	
	unsigned long	bitsize,
					infosize;
	BITMAPFILEHEADER	header;	

	if((fp = fopen(filename, "rb")) == NULL)
		return (NULL);

	if (fread(&header, sizeof(BITMAPFILEHEADER), 1, fp) < 1)
	{
		fclose(fp);
		return (NULL);
	}

	if(header.bfType != 'MB')	/* Check for BM reversed... */
	{
		fclose(fp);
		return (NULL);
	}

	infosize = header.bfOffBits - sizeof(BITMAPFILEHEADER);
	if((*info = (BITMAPINFO *)malloc(infosize)) == NULL)
	{
		fclose(fp);
		return (NULL);
	}

	if(fread(*info, 1, infosize, fp) < infosize)
	{
		free(*info);
		fclose(fp);
		return (NULL);
	}

	if((bitsize = (*info)->bmiHeader.biSizeImage) == 0)
		bitsize = ((*info)->bmiHeader.biWidth *
				   (*info)->bmiHeader.biBitCount + 7) / 8 *
  					abs((*info)->bmiHeader.biHeight);

	if((bits = (unsigned char *)malloc(bitsize)) == NULL)
	{
	   free(*info);
		fclose(fp);
		return (NULL);
	}

	if(fread(bits, 1, bitsize, fp) < bitsize)
	{
	   free(*info);
		free(bits);
		fclose(fp);
		return (NULL);
	}

	fclose(fp);
	return (bits);
}

void RRBMPImage::convertRGB(BITMAPINFO *info,	
           unsigned char *bits,	
		   unsigned char *newbits,
		   int BMPtoRGB)
{
	assert(info->bmiHeader.biCompression==BI_RGB);
  	assert(info->bmiHeader.biBitCount==24 || info->bmiHeader.biBitCount==8);

	const unsigned int bmpChan = info->bmiHeader.biBitCount/8; //8->1, 24->3
	const unsigned int rgbChan = 3;

	unsigned int widthByByte,newWidthByByte;
	unsigned int fromChan,toChan;
	if(BMPtoRGB)
	{
		unsigned int widthByPixel=info->bmiHeader.biWidth;
		newWidthByByte=rgbChan*widthByPixel;
		widthByByte=bmpChan*widthByPixel;	
		widthByByte=(widthByByte+3)& ~3; // padding to long int length
		fromChan=bmpChan;
		toChan=rgbChan;
	}
	else
	{
		unsigned int widthByPixel=info->bmiHeader.biWidth;
		widthByByte=rgbChan*widthByPixel;
		newWidthByByte=bmpChan*widthByPixel;
		newWidthByByte=(newWidthByByte+3) & ~3;	// padding to long int length
		fromChan=rgbChan;
		toChan=bmpChan;
	}

	unsigned char *from, *to;
	for(int i=0;i<info->bmiHeader.biHeight;i++)
	{
		from=bits;
		from+=i*widthByByte;
		to=newbits;
		to+=i*newWidthByByte;
		for(int j=0;
			j<info->bmiHeader.biWidth;
			j++,from+=fromChan,to+=toChan)
		{
			if(fromChan==1)
			{
				to[0] = from[0];
				to[1] = from[0];
				to[2] = from[0];
			}
			else if(toChan==1)
			{
				to[0] = from[2];
			}
			else
			{	// invert channel order
				to[0] = from[2];
				to[1] = from[1];
				to[2] = from[0];
			}
		}
	}
}

int RRBMPImage::saveDIBitmap(const char *filename,
	             BITMAPINFO *info,	
                 void       *bits)	
{
	FILE *fp;	
	if((fp=fopen(filename,"wb"))==NULL)
		return(-1);

	unsigned long bitsize;
	if(info->bmiHeader.biSizeImage == 0)
		bitsize=(info->bmiHeader.biWidth*info->bmiHeader.biBitCount+7)
	            /8*abs(info->bmiHeader.biHeight);
	else
		bitsize=info->bmiHeader.biSizeImage;

	unsigned long infosize=sizeof(BITMAPINFOHEADER);
	switch(info->bmiHeader.biCompression)
	{
	  case BI_BITFIELDS:
		infosize+=12;		
        if(info->bmiHeader.biClrUsed==0)
		break;
      case BI_RGB:
        if(info->bmiHeader.biBitCount>8 &&
           info->bmiHeader.biClrUsed==0)
		break;
      case BI_RLE8:
      case BI_RLE4:
        if(info->bmiHeader.biClrUsed==0)
			infosize+=(1 << info->bmiHeader.biBitCount)*4;
		else
			infosize+=info->bmiHeader.biClrUsed*4;
		break;
	}

	unsigned long size=sizeof(BITMAPFILEHEADER)+infosize+bitsize;

    BITMAPFILEHEADER header;	
	header.bfType      = 'MB';	
	header.bfSize      = size;
	header.bfReserved1 = 0;
	header.bfReserved2 = 0;
	header.bfOffBits   = sizeof(BITMAPFILEHEADER)+infosize;

	if(fwrite(&header,1,sizeof(BITMAPFILEHEADER),fp)<sizeof(BITMAPFILEHEADER))
	{
		fclose(fp);
		return (-1);
	}

	if(fwrite(info,1,infosize,fp)<infosize)
	{
		fclose(fp);
		return (-1);
	}

	if(fwrite(bits,1,bitsize,fp)<bitsize)
	{
		fclose(fp);
		return (-1);
	}

	fclose(fp);
	return (0);
}

RRStatus RRBMPImage::save()
{
	int xs,ys,ps;
	getSize(xs,ys);

	const int numPlane = 3;
	ps=getPixelByteSize(numPlane);
	long bitsize=xs*ys*ps;
	unsigned char *bits = new unsigned char[bitsize]; 
	getRawBits(numPlane,bits);

	BITMAPINFO *info = new BITMAPINFO;
	info->bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
	info->bmiHeader.biWidth         = xs;
	info->bmiHeader.biHeight        = ys;
	info->bmiHeader.biPlanes        = 1;
	info->bmiHeader.biBitCount      = ps*8;
	info->bmiHeader.biCompression   = BI_RGB;
	info->bmiHeader.biSizeImage     = bitsize;
	info->bmiHeader.biXPelsPerMeter = 2952; /* 75 DPI */
	info->bmiHeader.biYPelsPerMeter = 2952; /* 75 DPI */
	info->bmiHeader.biClrUsed       = 0;
	info->bmiHeader.biClrImportant  = 0;

	int width=3*info->bmiHeader.biWidth;
    width=(width+3) & ~3;	
    long cbitsize=width*info->bmiHeader.biHeight;
	unsigned char *cbits = new unsigned char[cbitsize];
	info->bmiHeader.biSizeImage     = cbitsize;
	convertRGB(info,bits,cbits,0);
	assert(cbits);

	if(saveDIBitmap(fileName, info, cbits)== -1)
		return RR_ERROR;

	free(info);
	free(bits);
	free(cbits);
	return RR_SUCCESS;
}

RRStatus RRBMPImage::load()
{
	BITMAPINFO *info;
	unsigned char *bits=loadDIBitmap(fileName,&info);
	if(bits==NULL)
		return RR_ERROR;

	unsigned char *cbits =
		new unsigned char[info->bmiHeader.biWidth*info->bmiHeader.biHeight*3];
	convertRGB(info,bits,cbits,1);
	assert(cbits);

//	dprintf("biSize          %d\n",info->bmiHeader.biSize);
//	dprintf("biWidth         %d\n",info->bmiHeader.biWidth);
//	dprintf("biHeight        %d\n",info->bmiHeader.biHeight);
//	dprintf("biPlanes        %d\n",info->bmiHeader.biPlanes);
//	dprintf("biBitCount      %d\n",info->bmiHeader.biBitCount);
//	dprintf("biCompression   %d\n",info->bmiHeader.biCompression);
//	dprintf("biSizeImage     %d\n",info->bmiHeader.biSizeImage);
//	dprintf("biXpelsPerMeter %d\n",info->bmiHeader.biXPelsPerMeter);
//	dprintf("biYPelsPerMeter %d\n",info->bmiHeader.biYPelsPerMeter);
//	dprintf("biClrUsed       %d\n",info->bmiHeader.biClrUsed);
//	dprintf("biClrImportant  %d\n",info->bmiHeader.biClrImportant);

	setSize(info->bmiHeader.biWidth,info->bmiHeader.biHeight);
	const int numPlane = 3;
	setRawBits(numPlane,cbits);
	free(info);
	free(bits);
	free(cbits);
	return RR_SUCCESS;
}

RRBMPImage::RRBMPImage(const char *filename)
:RRImage< RRPixelB >()
{
	strcpy(fileName,filename);
}

RRBMPImage::RRBMPImage(const char *filename, const RRImage< RRPixelB >& v)
:RRImage< RRPixelB >(v)
{
	strcpy(fileName,filename);
}

RRBMPImage::~RRBMPImage(){}

const RRBMPImage& RRBMPImage::operator=(const RRBMPImage& v)
{
	RRImage< RRPixelB >::operator=(v);
	//strcpy(fileName,v.fileName);
	return *this;
}

bool operator==(const RRBMPImage& v1,const RRBMPImage& v2)
{
	return (operator==((const RRImage< RRPixelB >&)v1,(const RRImage< RRPixelB >&)v2) &&
			!strcmp(v1.fileName,v2.fileName))?true:false;
}

bool operator!=(const RRBMPImage& v1,const RRBMPImage& v2)
{
	return (operator!=((const RRImage< RRPixelB >&)v1,(const RRImage< RRPixelB >&)v2) ||
			strcmp(v1.fileName,v2.fileName))?true:false;
}

//-------------------------------------------------------------------=72
// End of RRBMPImage.cpp
//-------------------------------------------------------------------=72
