//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RRBMPImage.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.2
//
// Modification History:
//  Oct/27/1998:	Copied from eldBMPImage.cpp
//	Nov/22/1998:	Renewed image and pixel classes
//	Apr/04/1999:	Including windows.h and wingdi.h is removed
//
// Bugs:
//	Correctly Works only on PC.  
//	
//-------------------------------------------------------------------=72

#include "RRBMPImage.h"

#include <stdio.h>
#include <string.h>

#define BI_RGB        0L
#define BI_RLE8       1L
#define BI_RLE4       2L
#define BI_BITFIELDS  3L

int RRWinBitmapFileHeader::SIZE_OF = 14;

unsigned char *RRBMPImage::mfLoadBitmap(
			FILE *fp, 
            RRWinBitmapInfoHeader&	info)
{
	RRWinBitmapFileHeader header;
	if(header.load(fp)==RR_ERROR)
		return NULL;

	if(header.bfType != 'MB')
	{
		fprintf(stderr,"RRBMPImage::mfLoadBitmap: bad magic number\n");
		return NULL;
	}

	const unsigned long offset = header.bfOffBits;
	
	if(info.load(fp)==RR_ERROR)
		return NULL;
		
	unsigned long bitsize=info.biSizeImage;
	if(bitsize==0)
		bitsize=(info.biWidth*info.biBitCount+7)/8*abs(info.biHeight);

	unsigned char *bits = new unsigned char[bitsize];	
	fread(bits,1,bitsize,fp);

	return bits;
}

RRStatus RRBMPImage::mfTransferBitmap(
			const RRWinBitmapInfoHeader& info,	
			unsigned char *bits,	
			unsigned char *newbits,
			bool BMPtoRGB)const
{
	if(info.biCompression!=BI_RGB ||
	   (info.biBitCount!=24 && info.biBitCount!=8))
		return RR_ERROR;

	const unsigned int bmpChan = info.biBitCount/8; //8->1, 24->3
	const unsigned int rgbChan = 3;

	unsigned int widthByByte,newWidthByByte;
	unsigned int fromChan,toChan;
	if(BMPtoRGB==true)
	{
		unsigned int widthByPixel=info.biWidth;
		newWidthByByte=rgbChan*widthByPixel;
		widthByByte=bmpChan*widthByPixel;	
		widthByByte=(widthByByte+3)& ~3; // padding to unsigned long int length
		fromChan=bmpChan;
		toChan=rgbChan;
	}
	else
	{
		unsigned int widthByPixel=info.biWidth;
		widthByByte=rgbChan*widthByPixel;
		newWidthByByte=bmpChan*widthByPixel;
		newWidthByByte=(newWidthByByte+3) & ~3;	// padding to unsigned long int length
		fromChan=rgbChan;
		toChan=bmpChan;
	}

	unsigned char *from, *to;
	for(unsigned int i=0;i<info.biHeight;i++)
	{
		from=bits;
		from+=i*widthByByte;
		to=newbits;
		to+=i*newWidthByByte;
		for(unsigned int j=0;
			j<info.biWidth;
			j++,from+=fromChan,to+=toChan)
		{
			if(fromChan==1)
			{
				to[0] = from[0];
				to[1] = from[0];
				to[2] = from[0];
			}
			else if(toChan==1)
			{
				to[0] = from[2];
			}
			else
			{	// invert channel order
				to[0] = from[2];
				to[1] = from[1];
				to[2] = from[0];
			}
		}
	}

	return RR_SUCCESS;
}

RRStatus RRBMPImage::mfSaveBitmap(
			FILE *fp,
	        const RRWinBitmapInfoHeader& info,	
            const void *bits)const	
{
	unsigned long bitsize;
	if(info.biSizeImage == 0)
		bitsize=(info.biWidth*info.biBitCount+7)/8*abs(info.biHeight);
	else
		bitsize=info.biSizeImage;

	unsigned long infosize=sizeof(RRWinBitmapInfoHeader);

	switch(info.biCompression)
	{
	  case BI_BITFIELDS:
		infosize+=12;		
        if(info.biClrUsed==0)
		break;
      case BI_RGB:
        if(info.biBitCount>8 && info.biClrUsed==0)
		break;
      case BI_RLE8:
      case BI_RLE4:
        if(info.biClrUsed==0)
			infosize+=(1 << info.biBitCount)*4;
		else
			infosize+=info.biClrUsed*4;
		break;
	}

	unsigned long size=RRWinBitmapFileHeader::SIZE_OF+infosize+bitsize;

    RRWinBitmapFileHeader header;	
	header.bfType = 'MB';	
	header.bfSize = size;
	header.bfReserved1 = 0;
	header.bfReserved2 = 0;
	header.bfOffBits = RRWinBitmapFileHeader::SIZE_OF+infosize;

	header.save(fp);
	info.save(fp);
	fwrite(bits,1,bitsize,fp);

	return RR_SUCCESS;
}

RRStatus RRBMPImage::save()const 
{
	int xs,ys,ps;
	getSize(xs,ys);

	const int numPlane = 3;
	ps=getPixelByteSize(numPlane);
	unsigned long bitsize=xs*ys*ps;
	unsigned char *bits = new unsigned char[bitsize]; 
	getRawBits(numPlane,bits);

	RRWinBitmapInfoHeader info;
	info.biSize = sizeof(RRWinBitmapInfoHeader);
	info.biWidth = xs;
	info.biHeight = ys;
	info.biPlanes = 1;
	info.biBitCount = ps*8;
	info.biCompression = BI_RGB;
	info.biSizeImage = bitsize;
	info.biXPelsPerMeter = 2952; /* 75 DPI */
	info.biYPelsPerMeter = 2952; /* 75 DPI */
	info.biClrUsed = 0;
	info.biClrImportant = 0;

	int width=3*info.biWidth;
    width=(width+3) & ~3;	
    unsigned long cbitsize=width*info.biHeight;
	unsigned char *cbits = new unsigned char[cbitsize];
	info.biSizeImage = cbitsize;
	if(mfTransferBitmap(info,bits,cbits,false)==RR_ERROR)
	{
		delete[] bits;
		delete[] cbits;
		return RR_ERROR;
	}

	FILE *fp=fopen(mFileName,"wb");
	if(fp==NULL)
	{
		delete[] bits;
		delete[] cbits;
		return RR_ERROR;
	}

	if(mfSaveBitmap(fp,info,cbits)==RR_ERROR)
		return RR_ERROR;

	fclose(fp);
	delete[] bits;
	delete[] cbits;
	return RR_SUCCESS;
}

RRStatus RRBMPImage::load()
{
	FILE *fp=fopen(mFileName,"rb");
	if(fp==NULL)
		return RR_ERROR;

	RRWinBitmapInfoHeader info;
	unsigned char *bits = mfLoadBitmap(fp,info);
	fclose(fp);

	if(bits==NULL)
		return RR_ERROR;

//	fprintf(stderr,"biSize          %d\n",info.biSize);
//	fprintf(stderr,"biWidth         %d\n",info.biWidth);
//	fprintf(stderr,"biHeight        %d\n",info.biHeight);
//	fprintf(stderr,"biPlanes        %d\n",info.biPlanes);
//	fprintf(stderr,"biBitCount      %d\n",info.biBitCount);
//	fprintf(stderr,"biCompression   %d\n",info.biCompression);
//	fprintf(stderr,"biSizeImage     %d\n",info.biSizeImage);
//	fprintf(stderr,"biXpelsPerMeter %d\n",info.biXPelsPerMeter);
//	fprintf(stderr,"biYPelsPerMeter %d\n",info.biYPelsPerMeter);
//	fprintf(stderr,"biClrUsed       %d\n",info.biClrUsed);
//	fprintf(stderr,"biClrImportant  %d\n",info.biClrImportant);

	unsigned char *cbits = new unsigned char[info.biWidth*info.biHeight*3];
	if(mfTransferBitmap(info,bits,cbits,true)==RR_ERROR)
	{
		delete[] cbits;
		delete[] bits;
		fclose(fp);
		return RR_ERROR;
	}

	setSize(info.biWidth,info.biHeight);
	const int numPlane = 3;
	setRawBits(numPlane,cbits);
	delete[] bits;
	delete[] cbits;
	return RR_SUCCESS;
}

RRBMPImage::RRBMPImage(const char *filename)
:RRImage< RRPixelB >()
{
	strcpy(mFileName,filename);
}

RRBMPImage::RRBMPImage(const char *filename, const RRImage< RRPixelB >& v)
:RRImage< RRPixelB >(v)
{
	strcpy(mFileName,filename);
}

RRBMPImage::~RRBMPImage(){}

const RRBMPImage& RRBMPImage::operator=(const RRBMPImage& v)
{
	RRImage< RRPixelB >::operator=(v);
	//strcpy(fileName,v.fileName);
	return *this;
}

bool operator==(const RRBMPImage& v1,const RRBMPImage& v2)
{
	return (operator==((const RRImage< RRPixelB >&)v1,(const RRImage< RRPixelB >&)v2) &&
			!strcmp(v1.mFileName,v2.mFileName))?true:false;
}

bool operator!=(const RRBMPImage& v1,const RRBMPImage& v2)
{
	return (operator!=((const RRImage< RRPixelB >&)v1,(const RRImage< RRPixelB >&)v2) ||
			strcmp(v1.mFileName,v2.mFileName))?true:false;
}

//-------------------------------------------------------------------=72
// End of RRBMPImage.cpp
//-------------------------------------------------------------------=72
