//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// RREventHandler.h
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Dec/05/1998:	Created	
//
// Bugs:
//
//-------------------------------------------------------------------=72
//
// RREventHandler class is the base class of which do event handling.
// A RREventHandler has a parent RREventHandler and one or more child 
// RREventHandlers.
// Each event is transmitted from parent to all its children by parent's 
// calling HandleEvent() of each child.
//
//-------------------------------------------------------------------=72

#ifndef _RR_EVENT_HANDLER_H
#define _RR_EVENT_HANDLER_H

#include "RRDef.h"
#include "RREvent.h"
#include "RRGrid2.h"
#include "RRArray.h"
#include "RRBoundingBox.h"

class RREventHandler
{
  private:
  
  	RREventHandler *mParent;
  	RRArray< RREventHandler* > mChild;
  	RREventHandler *mLocking;

  	RRBoundingBox mBBox;
  
  private:
  	 
	RRStatus mfHandleEventByChildren(
			const RREvent& e,
	        RREventHandler **hit);
  
  protected:
  
	RREventHandler *mfGetParent()const{return mParent;}
  	RREventHandler *mfGetChild(int i)const{return mChild[i];}
  	int mfGetNumChildren()const{return mChild.getSize();}
  	void mfSetLocking(RREventHandler *l){mLocking=l;}
  	const RREventHandler *mfGetLocking()const{return mLocking;}

	RRBoundingBox& mfGetBBox(){return mBBox;}
	const RRBoundingBox& mfGetBBox()const{return mBBox;}

  	virtual RRStatus mfHandleEvent(const RREvent& e) = 0;
  	// Called in handleEvent().
  	
 public:
  
	virtual RRStatus handleEvent(const RREvent& e);
  	// Processes the given event.
  	// The return value is ...
  	// RRSUCCESS if this RREventHandler processed the event.
  	// RRERROR if this RREventHandler didn't process the event.
  	// RRLOCKED if this RREventHandler processed and locked the following events.
  	// RRUNLOCKED if this RREventHandler processed and release its event lock.
  	
  	virtual int addChild(RREventHandler *c);
  	// Adds c to child, then return the number of children.
  	
  	virtual int removeChild(RREventHandler *c);
  	// Removes the RREventHandler* which is same to c from child, 
  	// then return the number of children.
  
  	RREventHandler(RREventHandler *pa);

	RREventHandler(
			RREventHandler *pa,
			const RRBoundingBox& bbox);
  	// Every RREventHandler must be constructed with a certain parent.
  	 
  	RREventHandler(const RREventHandler& x);
  	// A copy constructor.
  	
  	virtual ~RREventHandler();
  	
  	const RREventHandler& operator=(const RREventHandler& x);

	const RRBoundingBox& getBBox()const{return mBBox;}
};
    
#endif // _RR_EVENT_HANDLER_H

//-------------------------------------------------------------------=72
// End of IABitmap.cpp
//-------------------------------------------------------------------=72
