//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// IATransParamEstimation.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Dec/05/1998:	Created	
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "IATransParamEstimation.h"

#include <math.h>
#include <stdio.h>

IATransParamEstimation::IATransParamEstimation(
			const RRArray< RRData* >& in,	// RRGridArrayData -> in[0]
											// RRGridArrayData -> in[1]
			RRData *out)	                // IATransformParameterData
:RRProcess("IATransParamEstimation",in,out)
{}

RRStatus IATransParamEstimation::doIt()
{
	RRGridArrayData& pary=(RRGridArrayData&)(mInput[0]->getSubstance());
	RRGridArrayData& qary=(RRGridArrayData&)(mInput[1]->getSubstance());
    IATransformParameterData& out=(IATransformParameterData&)(mOutput->getSubstance());

	const RRArray< RRGrid2 >& p=pary.getGridArray();
	const RRArray< RRGrid2 >& q=qary.getGridArray();
	if(p.getSize()!=q.getSize())
		return RR_ERROR;

	double rot,tx,ty;
	if(mfCalcParameters(p,q,rot,tx,ty)==RR_ERROR)
		return RR_ERROR;

	out.setRotation(rot);
	out.setTranslationX(tx);
	out.setTranslationY(ty);
	return RR_SUCCESS;
}

RRStatus IATransParamEstimation::mfCalcParameters(
			const RRArray< RRGrid2 >& p,
			const RRArray< RRGrid2 >& q,
			double& rot,
			double& tx,
			double& ty)
//
// compute transform: p -> q
// by least square estimation
//  
// qx = px*cos(rot) - py*sin(rot) + tx 
// qy = px*sin(rot) + py*cos(rot) + ty 
//
{
	int numdata=p.getSize();
	double spx(0),spy(0),sqx(0),sqy(0),spxqx(0),spxqy(0),spyqx(0),spyqy(0);
	for(int i=0;i<numdata;i++)
	{
		spx+=p[i].getX();
		spy+=p[i].getY();
		sqx+=q[i].getX();
		sqy+=q[i].getY();
		spxqx+=p[i].getX()*q[i].getX();
		spxqy+=p[i].getX()*q[i].getY();
		spyqx+=p[i].getY()*q[i].getX();
		spyqy+=p[i].getY()*q[i].getY();
	}
	
	double sinr=(spy*sqx-spx*sqy)/numdata-spyqx+spxqy;
	double cosr=(-spx*sqx-spy*sqy)/numdata+spxqx+spyqy;
	rot=atan2(sinr,cosr);
	tx=(sqx-spx*cos(rot)+spy*sin(rot))/numdata;
	ty=(sqy-spx*sin(rot)-spy*cos(rot))/numdata;
	
	return RR_SUCCESS;
}

//-------------------------------------------------------------------=72
// End of IATransParamEstimation.cpp
//-------------------------------------------------------------------=72

