//-----------------------------------------------------------
// IALoadSaveImage.C
//-----------------------------------------------------------

#include "IALoadSaveImage.h"
#include "RRBMPImage.h"
#include "RRPPMImage.h"
#include "RRPPM2Image.h"
#include "PFMBinaryFile.h"

#include <string.h>

const char *fileExtension(const char *filename)
{
	return strrchr(filename,'.');
}

RRStatus loadBMPToFiRR(
			const char *imagefile,
			RRFiRR< RRPixelD >& fld)
{
	RRROBMPImage img(imagefile);
	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,img.getPixelD(x,y));
	return RRSUCCESS;	
}

RRStatus saveBMPFromFiRR(
			const RRFiRR< RRPixelD >& fld,
			const char *imagefile)
{
	RRWOBMPImage img(imagefile);
	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setPixelD(x,y,fld.getElem(x,y));
	return RRSUCCESS;	
}

RRStatus loadPPMToFiRR(
			const char *imagefile,
			RRFiRR< RRPixelD >& fld)
{
	RRROPPMImage img(imagefile);
	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,img.getPixelD(x,y));
	return RRSUCCESS;	
}

RRStatus savePPMFromFiRR(
			const RRFiRR< RRPixelD >& fld,
			const char *imagefile)
{
	RRWOPPMImage img(imagefile);
	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setPixelD(x,y,fld.getElem(x,y));
	return RRSUCCESS;	
}

RRStatus loadPPM2ToFiRR(
			const char *imagefile,
			RRFiRR< RRPixelD >& fld)
{
	RRROPPM2Image img(imagefile);
	int xs,ys;
	img.getSize(xs,ys);
	fld.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			fld.setElem(x,y,img.getPixelD(x,y));
	return RRSUCCESS;	
}

RRStatus savePPM2FromFiRR(
			const RRFiRR< RRPixelD >& fld,
			const char *imagefile)
{
	RRWOPPM2Image img(imagefile);
	int xs,ys;
	fld.getSize(xs,ys);
	img.setSize(xs,ys);

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
			img.setPixelD(x,y,fld.getElem(x,y));
	return RRSUCCESS;	
}

RRStatus loadPFMToFiRR(
			const char *imagefile,
			RRFiRR< RRPixelD >& fld)
{
	ROPFMBinaryFile img(imagefile);
	int xs,ys,scale;
	img.readHeader(xs,ys,scale);
	fld.setSize(xs,ys);

	for(int y=ys-1;y>=0;y--)
		for(int x=0;x<xs;x++)
		{
			float r,g,b;
			img.readData(&r);
			img.readData(&g);
			img.readData(&b);
			RRPixelD p(double(r/scale),double(g/scale),double(b/scale),1.);
			fld.setElem(x,y,p);
		}
	return RRSUCCESS;	
}

RRStatus savePFMFromFiRR(
			const RRFiRR< RRPixelD >& fld,
			const char *imagefile)
{
	const int scale = 255;
	int xs,ys;
	fld.getSize(xs,ys);
	WOPFMBinaryFile img(imagefile);
	img.writeHeader(xs,ys,scale);

	for(int y=ys-1;y>=0;y--)
		for(int x=0;x<xs;x++)
		{
			RRPixelD p=fld.getElem(x,y);
			float r=float(p.getR())*scale;
			float g=float(p.getG())*scale;
			float b=float(p.getB())*scale;
			img.writeData(&r);
			img.writeData(&g);
			img.writeData(&b);
		}
	return RRSUCCESS;	
}

RRStatus loadImageToFiRR(
			const char *imagefile,
			RRFiRR< RRPixelD >& fld)
{
	if(strcmp(fileExtension(imagefile),".bmp")==0)
	{
		return loadBMPToFiRR(imagefile,fld);
	}
	else if(strcmp(fileExtension(imagefile),".ppm")==0)
	{
		return loadPPMToFiRR(imagefile,fld);
	}
	else if(strcmp(fileExtension(imagefile),".ppm2")==0)
	{
		return loadPPM2ToFiRR(imagefile,fld);
	}
	else if(strcmp(fileExtension(imagefile),".pfm")==0)
	{
		return loadPFMToFiRR(imagefile,fld);
	}
	else
		return RRERROR;
}

RRStatus saveImageFromFiRR(
			const RRFiRR< RRPixelD >& fld,
			const char *imagefile)
{
	if(strcmp(fileExtension(imagefile),".bmp")==0)
	{
		return saveBMPFromFiRR(fld,imagefile);
	}
	else if(strcmp(fileExtension(imagefile),".ppm")==0)
	{
		return savePPMFromFiRR(fld,imagefile);
	}
	else if(strcmp(fileExtension(imagefile),".ppm2")==0)
	{
		return savePPM2FromFiRR(fld,imagefile);
	}
	else if(strcmp(fileExtension(imagefile),".pfm")==0)
	{
		return savePFMFromFiRR(fld,imagefile);
	}
	else
		return RRERROR;
}

//-----------------------------------------------------------
// End of IALoadSaveImage.C
//-----------------------------------------------------------
