//-------------------------------------------------------------------=72
//
// Copyright (C) Columbia University, 1998-1999. All Rights Reserved.
//
//-------------------------------------------------------------------=72
//
// IAImageTransform.cpp
//
//-------------------------------------------------------------------=72
//
// Author:				 Tomoo Mitsunaga
//
// Version:              1.0
//
// Modification History:
//  Dec/05/1998:	Created	
//
// Bugs:
//
//-------------------------------------------------------------------=72

#include "IAImageTransform.h"
#include "RRImageFileIO.h"

#include <math.h>
#include <stdio.h>

IAImageTransform::IAImageTransform(
			const RRArray< RRData* >& in,	// RRFileListData -> in[0]
											// IATransformParameterData -> in[1]
			RRData *out)	                // RRFileListData
:RRProcess("IAImageTransform",in,out)
{}

RRStatus IAImageTransform::doIt()
{
	RRFileListData& sdata=(RRFileListData&)(mInput[0]->getSubstance());
	IATransformParameterData& pdata=(IATransformParameterData&)(mInput[1]->getSubstance());
    RRFileListData& tdata=(RRFileListData&)(mOutput->getSubstance());

	RRField< RRPixelD > src;
	if(RRImageFileIO::load(sdata.getFileName(0),RRIFIO_EXT,RRIFIO_PIXELD,src)==RR_ERROR)
		return RR_ERROR;

	RRField< RRPixelD > trans;
	int xs,ys;
	src.getSize(xs,ys);
	trans.setSize(xs,ys);

	double rot=pdata.getRotation();
	double tx=pdata.getTranslationX();
	double ty=pdata.getTranslationY();

	for(int y=0;y<ys;y++)
		for(int x=0;x<xs;x++)
		{
			double px=x*cos(rot)-y*sin(rot)+tx;
			double py=x*sin(rot)+y*cos(rot)+ty;
		
			trans.setElem(x,y,mfResampleSinc(src,px,py));
		}

	if(RRImageFileIO::save(trans,RRIFIO_PIXELD,RRIFIO_EXT,tdata.getFileName(0))==RR_ERROR)
		return RR_ERROR;

	return RR_SUCCESS;
}

RRPixelD IAImageTransform::mfResampleBilinear(
			const RRField< RRPixelD >& img,
			const double x,
			const double y)
{ // bilinear interpolation
	int lx=int(x);
	int ux=lx+1;
	int ly=int(y);
	int uy=ly+1;

	int xs,ys;
	img.getSize(xs,ys);
	RRPixelD pbl=(lx<0 || 
	               lx>=xs || 
				   ly <0 || 
				   ly >=ys)?RRPixelD(0,0,0,1):
				            img.getElem(lx,ly);
	RRPixelD pbr=(ux<0 || 
	               ux>=xs || 
				   ly <0 || 
				   ly >=ys)?RRPixelD(0,0,0,1):
				            img.getElem(ux,ly);
	RRPixelD ptl=(lx<0 || 
	               lx>=xs || 
				   uy <0 || 
				   uy >=ys)?RRPixelD(0,0,0,1):
				            img.getElem(lx,uy);
	RRPixelD ptr=(ux<0 || 
	               ux>=xs || 
				   uy <0 || 
				   uy >=ys)?RRPixelD(0,0,0,1):
				            img.getElem(ux,uy);

	double rbl=(ux-x)*(uy-y);
	double rbr=(x-lx)*(uy-y);
	double rtl=(ux-x)*(y-ly);
	double rtr=(x-lx)*(y-ly);

	RRPixelD ret;
	ret.setR(rbl*pbl.getR()+rbr*pbr.getR()+rtl*ptl.getR()+rtr*ptr.getR());
	ret.setG(rbl*pbl.getG()+rbr*pbr.getG()+rtl*ptl.getG()+rtr*ptr.getG());
	ret.setB(rbl*pbl.getB()+rbr*pbr.getB()+rtl*ptl.getB()+rtr*ptr.getB());
	return ret;
}

double IAImageTransform::mfSinc(
			const double x)
{
	if(x==0)
		return 1.;

	double xpi=x*M_PI;
	return sin(xpi)/xpi;

	// test code of nearest neighbor
	// return (fabs(x)<0.5)?1:0;
}

RRPixelD IAImageTransform::mfResampleSinc(
			const RRField< RRPixelD >& img,
			const double x,
			const double y)
{ // interpolation by Sinc(x)
	const int kernelSize = 4; // must be even

	int lx=int(x)-(kernelSize/2-1);
	int ux=lx+(kernelSize-1);
	int ly=int(y)-(kernelSize/2-1);
	int uy=ly+(kernelSize-1);

	RRArray< double > wx(kernelSize);
	RRArray< double > wy(kernelSize);
	for(int i=0;i<kernelSize;i++)
	{
		wx[i]=mfSinc(lx+i-x);
		wy[i]=mfSinc(ly+i-y);
	}

	int xs,ys;
	img.getSize(xs,ys);

	double sw(0.);
	double r(0.),g(0.),b(0.);
	for(int iy=ly;iy<=uy;iy++)
		for(int ix=lx;ix<=ux;ix++)
		{
			double w=wx[ix-lx]*wy[iy-ly];
			sw+=w;

			if(ix<0 || ix>=xs || iy<0 || iy >=ys)
				continue;
			
			RRPixelD p=img.getElem(ix,iy);
			r+=p.getR()*w; 
			g+=p.getG()*w; 
			b+=p.getB()*w; 
		}

	return RRPixelD(r/sw,g/sw,b/sw,1.);
}

//-------------------------------------------------------------------=72
// End of IAImageTransform.cpp
//-------------------------------------------------------------------=72

